# Store Category Assignment Fix

## Problem Identified

Your stores are showing in the wrong categories because of incorrect database assignments. Specifically:

- **Blackberrys** (Fashion brand) is assigned to category 1 (Beauty & Health) instead of category 2 (Fashion)
- **Go Colors** (Fashion brand) is also in Beauty & Health instead of Fashion
- Many other stores are clustered in Beauty & Health when they should be in other categories

## Root Cause

The system uses TWO methods to assign stores to categories:

1. **Legacy Field**: `stores.store_category` (single category ID)
2. **Junction Table**: `store_categories` (allows multiple categories)

Both need to be updated when changing a store's category.

## Category IDs

- **1** = Beauty & Health
- **2** = Fashion
- **4** = Travel
- **7** = Electronics/Accessories
- **15** = Services/Telecom
- (and more...)

## Solution Options

### Option 1: Use the PHP Fix Tool (Recommended)

1. Upload `fix_store_categories.php` to your website root
2. Access it in your browser: `https://yourdomain.com/fix_store_categories.php`
3. Review the store list and category assignments
4. Click "Fix All Fashion Stores" to automatically fix known fashion brands
5. Or manually fix individual stores using the form at the bottom

### Option 2: Run SQL Script Directly

Execute the `fix_store_categories.sql` file in your database:

```sql
-- Fix Blackberrys
UPDATE stores SET store_category = 2 WHERE store_id = 6;
DELETE FROM store_categories WHERE store_id = 6;
INSERT INTO store_categories (store_id, category_id) VALUES (6, 2);

-- Fix Go Colors
UPDATE stores SET store_category = 2 WHERE store_id = 21;
DELETE FROM store_categories WHERE store_id = 21;
INSERT INTO store_categories (store_id, category_id) VALUES (21, 2);
```

### Option 3: Fix via Admin Panel

1. Go to Admin → Stores
2. Edit "Blackberrys"
3. Change "Parent Category" to "Fashion"
4. Save
5. Repeat for other misassigned stores

## Stores That Need Fixing

Based on the database analysis, these stores are currently in wrong categories:

| Store Name | Store ID | Current Category | Should Be | Reason |
|------------|----------|------------------|-----------|---------|
| Blackberrys | 6 | Beauty & Health (1) | Fashion (2) | Fashion/Clothing brand |
| Go Colors | 21 | Beauty & Health (1) | Fashion (2) | Fashion/Clothing brand |
| Montecarlo | 40 | Beauty & Health (1) | Fashion (2) | Fashion/Clothing brand |
| House of Masaba | 106 | Beauty & Health (1) | Fashion (2) | Fashion/Clothing brand |
| Boat | 99 | Fashion (2) | Electronics (7) | Electronics/Audio accessories |
| Innisfree | 100 | Fashion (2) | Beauty & Health (1) | Beauty/Skincare brand |

## How to Prevent This Issue

When adding or editing stores in the admin panel:

1. Always select the correct **Parent Category** first
2. Optionally select a **Subcategory** if applicable
3. The system will automatically update both the legacy field and junction table
4. Verify the store appears in the correct category on the frontend

## Verification

After applying fixes, verify:

1. Visit your Fashion category page
2. Check that Blackberrys and Go Colors appear there
3. Visit Beauty & Health category
4. Confirm they no longer appear there
5. Check the store count on homepage categories

## Technical Details

The category filtering query in `functions.php` checks BOTH assignment methods:

```php
WHERE (
    CAST(stores.store_category AS UNSIGNED) = :item 
    OR EXISTS (
        SELECT 1 FROM store_categories sc 
        WHERE sc.store_id = stores.store_id 
        AND sc.category_id = :item
    )
)
```

This means if EITHER the legacy field OR the junction table has the wrong category, the store will appear in the wrong section.

## Files Involved

- `functions.php` - Category filtering logic
- `admin/controller/edit_store.php` - Store update handler
- `admin/views/edit.store.view.php` - Store edit form
- Database tables: `stores`, `store_categories`, `categories`
