<?php
/**
 * Patch Admin Functions - Add Missing Product Functions
 * Run this ONCE to add product functions to admin/functions.php
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<h1>Patch Admin Functions</h1>";

$functionsFile = 'admin/functions.php';

if (!file_exists($functionsFile)) {
    die("<p style='color:red;'>Error: admin/functions.php not found!</p>");
}

// Read current content
$content = file_get_contents($functionsFile);

// Check if already patched
if (strpos($content, 'function getProductById') !== false) {
    echo "<p style='color:orange;'>⚠ Already patched! getProductById function already exists.</p>";
    echo "<p>No changes needed.</p>";
    exit;
}

// Product functions to add
$productFunctions = "
// PRODUCTS ---------------------------------------

/**
 * Get all products
 */
function get_all_products(\$connect){
    \$sql = \"SELECT p.*, s.store_title, c.category_title 
            FROM products p
            LEFT JOIN stores s ON p.product_store = s.store_id
            LEFT JOIN categories c ON p.product_category = c.category_id
            ORDER BY p.product_id DESC\";
    \$sentence = \$connect->prepare(\$sql); 
    \$sentence->execute();
    return \$sentence->fetchAll();
}

/**
 * Get single product by ID
 */
function getProductById(\$connect, \$product_id) {
    \$sentence = \$connect->prepare(\"SELECT p.*, s.store_title, s.store_slug, c.category_title, c.category_slug 
                                    FROM products p
                                    LEFT JOIN stores s ON p.product_store = s.store_id
                                    LEFT JOIN categories c ON p.product_category = c.category_id
                                    WHERE p.product_id = :product_id\");
    \$sentence->execute([':product_id' => \$product_id]);
    return \$sentence->fetch(PDO::FETCH_ASSOC);
}

/**
 * Count products by store
 */
function countProductsByStore(\$connect, \$store_id) {
    \$sentence = \$connect->prepare(\"SELECT COUNT(*) as total FROM products WHERE product_store = :store_id AND product_status = 1\");
    \$sentence->execute([':store_id' => \$store_id]);
    \$result = \$sentence->fetch(PDO::FETCH_ASSOC);
    return \$result['total'];
}

/**
 * Count products by category
 */
function countProductsByCategory(\$connect, \$category_id) {
    \$sentence = \$connect->prepare(\"SELECT COUNT(*) as total FROM products WHERE product_category = :category_id AND product_status = 1\");
    \$sentence->execute([':category_id' => \$category_id]);
    \$result = \$sentence->fetch(PDO::FETCH_ASSOC);
    return \$result['total'];
}

?>";

// Remove closing ?> if it exists at the end
$content = rtrim($content);
if (substr($content, -2) === '?>') {
    $content = substr($content, 0, -2);
}

// Add product functions
$newContent = $content . "\n" . $productFunctions;

// Backup original file
$backupFile = 'admin/functions.php.backup.' . date('YmdHis');
if (copy($functionsFile, $backupFile)) {
    echo "<p style='color:green;'>✓ Backup created: $backupFile</p>";
} else {
    echo "<p style='color:red;'>✗ Could not create backup!</p>";
    exit;
}

// Write new content
if (file_put_contents($functionsFile, $newContent)) {
    echo "<p style='color:green;'><strong>✓✓✓ SUCCESS! ✓✓✓</strong></p>";
    echo "<p>Product functions added to admin/functions.php</p>";
    echo "<p>Functions added:</p>";
    echo "<ul>";
    echo "<li>get_all_products()</li>";
    echo "<li>getProductById()</li>";
    echo "<li>countProductsByStore()</li>";
    echo "<li>countProductsByCategory()</li>";
    echo "</ul>";
    
    echo "<hr>";
    echo "<h2>Next Steps:</h2>";
    echo "<ol>";
    echo "<li>Test edit product: <a href='admin/controller/edit_product.php?id=11'>Edit Product</a></li>";
    echo "<li>If it works, you're done!</li>";
    echo "<li>If not, check the error message</li>";
    echo "</ol>";
    
    echo "<hr>";
    echo "<p><strong>For Blog:</strong> Clear Cloudflare cache or use <a href='blog_working.php'>blog_working.php</a></p>";
    
} else {
    echo "<p style='color:red;'>✗ Error: Could not write to admin/functions.php</p>";
    echo "<p>Check file permissions (should be 644 or 755)</p>";
}
?>
