<?php
/**
 * Import Stores from CSV
 * Prevents duplicate imports based on store title
 */

require "core.php";

set_time_limit(300); // 5 minutes

$csvFile = 'stores.csv';

if (!file_exists($csvFile)) {
    die("Error: stores.csv file not found!");
}

echo "<!DOCTYPE html><html><head><title>Import Stores</title>";
echo "<style>
body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
.container { max-width: 1000px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; }
h1 { color: #333; border-bottom: 3px solid #ed174f; padding-bottom: 10px; }
.success { color: #28a745; padding: 5px 0; }
.skip { color: #ffc107; padding: 5px 0; }
.error { color: #dc3545; padding: 5px 0; }
.summary { background: #e7f3ff; padding: 15px; border-left: 4px solid #2196F3; margin: 20px 0; }
.summary h2 { margin-top: 0; color: #2196F3; }
</style></head><body><div class='container'>";

echo "<h1>📦 Importing Stores from CSV</h1>";

// Read CSV file
$file = fopen($csvFile, 'r');
$header = fgetcsv($file); // Skip header row

$imported = 0;
$skipped = 0;
$errors = 0;

echo "<div style='max-height: 400px; overflow-y: auto; border: 1px solid #ddd; padding: 10px; margin: 20px 0;'>";

while (($row = fgetcsv($file)) !== false) {
    $storeName = trim($row[0]);
    
    // Skip empty rows
    if (empty($storeName)) {
        continue;
    }
    
    try {
        // Check if store already exists (by title or slug)
        $slug = strtolower(str_replace(' ', '-', $storeName));
        $slug = preg_replace('/[^a-z0-9-]/', '', $slug);
        
        $checkStmt = $connect->prepare("SELECT store_id FROM stores WHERE store_title = :title OR store_slug = :slug LIMIT 1");
        $checkStmt->execute([
            ':title' => $storeName,
            ':slug' => $slug
        ]);
        
        if ($checkStmt->fetch()) {
            echo "<div class='skip'>⚠ Skipped: <strong>{$storeName}</strong> (already exists)</div>";
            $skipped++;
            continue;
        }
        
        // Insert new store
        $insertStmt = $connect->prepare("
            INSERT INTO stores (
                store_title,
                store_slug,
                store_description,
                store_image,
                store_featured,
                store_status
            ) VALUES (
                :title,
                :slug,
                :description,
                'default-store.png',
                0,
                1
            )
        ");
        
        $insertStmt->execute([
            ':title' => $storeName,
            ':slug' => $slug,
            ':description' => "Get the best deals and coupons for {$storeName}. Save money on your purchases with exclusive offers."
        ]);
        
        echo "<div class='success'>✓ Imported: <strong>{$storeName}</strong></div>";
        $imported++;
        
    } catch (Exception $e) {
        echo "<div class='error'>✗ Error importing <strong>{$storeName}</strong>: " . $e->getMessage() . "</div>";
        $errors++;
    }
}

fclose($file);

echo "</div>";

// Summary
echo "<div class='summary'>";
echo "<h2>📊 Import Summary</h2>";
echo "<p><strong>Total Imported:</strong> {$imported} stores</p>";
echo "<p><strong>Skipped (Duplicates):</strong> {$skipped} stores</p>";
echo "<p><strong>Errors:</strong> {$errors} stores</p>";
echo "<p><strong>Total Processed:</strong> " . ($imported + $skipped + $errors) . " stores</p>";
echo "</div>";

echo "<div style='margin-top: 20px;'>";
echo "<a href='admin/controller/stores.php' style='display: inline-block; padding: 10px 20px; background: #ed174f; color: white; text-decoration: none; border-radius: 5px;'>View Stores in Admin</a>";
echo "</div>";

echo "</div></body></html>";
?>
