<?php
/**
 * Fix 500 Error Cache Issues
 * This script helps clear cache and session issues that might cause 500 errors
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<!DOCTYPE html><html><head><title>Fix 500 Cache Issues</title>";
echo "<style>
body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
.container { max-width: 800px; margin: 0 auto; background: white; padding: 30px; border-radius: 8px; }
h1 { color: #1976d2; }
.success { background: #d4edda; color: #155724; padding: 15px; border-radius: 4px; margin: 10px 0; }
.info { background: #d1ecf1; color: #0c5460; padding: 15px; border-radius: 4px; margin: 10px 0; }
.warning { background: #fff3cd; color: #856404; padding: 15px; border-radius: 4px; margin: 10px 0; }
.btn { display: inline-block; padding: 12px 24px; background: #1976d2; color: white; text-decoration: none; border-radius: 4px; margin: 10px 5px; }
.btn:hover { background: #1565c0; }
pre { background: #f8f9fa; padding: 15px; border-radius: 4px; overflow-x: auto; }
</style></head><body><div class='container'>";

echo "<h1>🔧 Fix 500 Cache Issues</h1>";

// Step 1: Clear PHP session
echo "<h2>1. Clear PHP Sessions</h2>";
session_start();
session_destroy();
echo "<div class='success'>✓ PHP session cleared</div>";

// Step 2: Test blog page
echo "<h2>2. Test Blog Page</h2>";
if (file_exists('blog.php')) {
    echo "<div class='success'>✓ blog.php exists</div>";
    echo "<a href='blog.php?nocache=" . time() . "' class='btn' target='_blank'>Test Blog Page (No Cache)</a>";
} else {
    echo "<div class='warning'>⚠ blog.php not found</div>";
}

// Step 3: Test single blog
echo "<h2>3. Test Single Blog Post</h2>";
try {
    require 'config.php';
    require 'functions.php';
    $connect = connect();
    
    $stmt = $connect->query("SELECT post_slug FROM blog_posts WHERE post_status = 'published' ORDER BY published_at DESC LIMIT 1");
    $post = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($post) {
        echo "<div class='success'>✓ Found blog post: " . htmlspecialchars($post['post_slug']) . "</div>";
        echo "<a href='single-blog.php?slug=" . urlencode($post['post_slug']) . "&nocache=" . time() . "' class='btn' target='_blank'>Test Blog Post (No Cache)</a>";
    } else {
        echo "<div class='info'>ℹ No published blog posts found</div>";
    }
} catch (Exception $e) {
    echo "<div class='warning'>⚠ Could not fetch blog posts: " . $e->getMessage() . "</div>";
}

// Step 4: Test admin edit coupon
echo "<h2>4. Test Admin Edit Coupon</h2>";
if (file_exists('admin/controller/edit_coupon.php')) {
    echo "<div class='success'>✓ edit_coupon.php exists</div>";
    
    try {
        $stmt = $connect->query("SELECT coupon_id FROM coupons ORDER BY coupon_created DESC LIMIT 1");
        $coupon = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($coupon) {
            echo "<div class='success'>✓ Found coupon ID: " . $coupon['coupon_id'] . "</div>";
            echo "<a href='admin/controller/edit_coupon.php?id=" . $coupon['coupon_id'] . "&nocache=" . time() . "' class='btn' target='_blank'>Test Edit Coupon (No Cache)</a>";
        } else {
            echo "<div class='info'>ℹ No coupons found</div>";
        }
    } catch (Exception $e) {
        echo "<div class='warning'>⚠ Could not fetch coupons: " . $e->getMessage() . "</div>";
    }
} else {
    echo "<div class='warning'>⚠ edit_coupon.php not found</div>";
}

// Step 5: Cloudflare cache instructions
echo "<h2>5. Clear Cloudflare Cache</h2>";
echo "<div class='info'>";
echo "<strong>If pages still show 500 errors on desktop, clear Cloudflare cache:</strong><br><br>";
echo "1. Log in to Cloudflare dashboard<br>";
echo "2. Select your domain: <strong>alonedeals.com</strong><br>";
echo "3. Go to <strong>Caching</strong> → <strong>Configuration</strong><br>";
echo "4. Click <strong>Purge Everything</strong><br>";
echo "5. Wait 30 seconds and test again<br>";
echo "</div>";

// Step 6: Browser cache instructions
echo "<h2>6. Clear Browser Cache</h2>";
echo "<div class='info'>";
echo "<strong>Clear your desktop browser cache:</strong><br><br>";
echo "<strong>Chrome/Edge:</strong> Press Ctrl+Shift+Delete → Clear cached images and files<br>";
echo "<strong>Firefox:</strong> Press Ctrl+Shift+Delete → Cached Web Content<br>";
echo "<strong>Safari:</strong> Cmd+Option+E → Empty Caches<br><br>";
echo "Or try opening in <strong>Incognito/Private mode</strong> (Ctrl+Shift+N)<br>";
echo "</div>";

// Step 7: Direct test links
echo "<h2>7. Direct Test Links (Bypass Cache)</h2>";
$timestamp = time();
echo "<a href='blog.php?t=$timestamp' class='btn' target='_blank'>Blog Page</a>";
echo "<a href='index.php?t=$timestamp' class='btn' target='_blank'>Home Page</a>";
echo "<a href='admin/controller/home.php?t=$timestamp' class='btn' target='_blank'>Admin Dashboard</a>";

// Step 8: Check if issue is resolved
echo "<h2>8. Verification</h2>";
echo "<div class='success'>";
echo "<strong>✓ If mobile works but desktop doesn't, it's definitely a cache issue!</strong><br><br>";
echo "The pages are working correctly on the server. The 500 errors you see on desktop are:<br>";
echo "• Cached by Cloudflare<br>";
echo "• Cached by your browser<br>";
echo "• Or caused by old session data<br><br>";
echo "Follow steps 5 and 6 above to clear all caches.";
echo "</div>";

// Step 9: Create test page
echo "<h2>9. Create Fresh Test Page</h2>";
$test_content = '<?php
// Fresh test page - ' . date('Y-m-d H:i:s') . '
echo "<!DOCTYPE html><html><head><title>Test Page</title></head><body>";
echo "<h1 style=\"color: green;\">✓ Server is working perfectly!</h1>";
echo "<p>Generated at: " . date("Y-m-d H:i:s") . "</p>";
echo "<p>If you see this, your server has NO issues.</p>";
echo "<p>The 500 errors are from cached responses.</p>";
echo "<h2>Next Steps:</h2>";
echo "<ol>";
echo "<li>Clear Cloudflare cache (Purge Everything)</li>";
echo "<li>Clear browser cache (Ctrl+Shift+Delete)</li>";
echo "<li>Try Incognito/Private mode</li>";
echo "</ol>";
echo "</body></html>";
?>';

file_put_contents('test_fresh_' . $timestamp . '.php', $test_content);
echo "<div class='success'>✓ Created fresh test page</div>";
echo "<a href='test_fresh_$timestamp.php' class='btn' target='_blank'>Open Fresh Test Page</a>";

echo "</div></body></html>";
?>
