# CDN Setup Instructions

## Overview
All redesign CSS and JS files are now centralized in 3 files for easy CDN URL management.

## 📁 Centralized Files Created:

### 1. **views/includes/header-css.php**
Contains all CSS files loaded in the `<head>` section:
- `filter-sidebar-redesign.css`
- `sidenav-redesign.css`
- `categories-slider-redesign.css`
- `homepage-featured-offers.css`
- `campaigns-redesign.css`
- `why-choose-us.css`
- `coupon-modal-redesign.css`

### 2. **views/includes/footer-css.php**
Contains CSS files loaded before `</body>`:
- `footer-redesign.css`
- `modern-footer-newsletter.css`

### 3. **views/includes/footer-js.php**
Contains JavaScript files loaded before `</body>`:
- `modern-footer-newsletter.js`

---

## 🚀 How to Use CDN URLs:

### Step 1: Upload CSS/JS Files to Your CDN

Upload these files to your CDN (Cloudflare R2, S3, etc.):

**CSS Files:**
```
assets/css/filter-sidebar-redesign.css
assets/css/sidenav-redesign.css
assets/css/categories-slider-redesign.css
assets/css/homepage-featured-offers.css
assets/css/campaigns-redesign.css
assets/css/why-choose-us.css
assets/css/coupon-modal-redesign.css
assets/css/footer-redesign.css
assets/css/modern-footer-newsletter.css
```

**JS Files:**
```
assets/js/modern-footer-newsletter.js
```

### Step 2: Update CDN URLs

Edit the 3 centralized files and set your CDN base URL:

#### **views/includes/header-css.php**
```php
// Line 8: Set your CDN base URL
$cdn_css_url = 'https://cdn.example.com/css';
```

#### **views/includes/footer-css.php**
```php
// Line 8: Set your CDN base URL
$cdn_css_url = 'https://cdn.example.com/css';
```

#### **views/includes/footer-js.php**
```php
// Line 8: Set your CDN base URL
$cdn_js_url = 'https://cdn.example.com/js';
```

### Step 3: Test

1. Clear browser cache
2. Visit your website
3. Check browser DevTools → Network tab
4. Verify CSS/JS files are loading from CDN URLs

---

## 📍 Where Files Are Included:

### Header CSS
**File:** `views/header.view.php`  
**Line:** ~95  
```php
<!-- Centralized Redesign CSS -->
<?php include __DIR__ . '/includes/header-css.php'; ?>
```

### Footer CSS & JS
**File:** `sections/views/footer.view.php`  
**Line:** ~150  
```php
<!-- Centralized Footer CSS & JS -->
<?php include __DIR__ . '/../includes/footer-css.php'; ?>
<?php include __DIR__ . '/../includes/footer-js.php'; ?>
```

---

## ✅ Benefits:

1. **Single Point of Control** - Change CDN URL in one place
2. **Easy Switching** - Toggle between local and CDN by changing one variable
3. **Version Control** - Add cache busting easily
4. **Clean Code** - No duplicate CSS links across files
5. **Fast Updates** - Update all pages at once

---

## 🔄 Switching Between Local and CDN:

### Use CDN:
```php
$cdn_css_url = 'https://cdn.example.com/css';
```

### Use Local Files:
```php
$cdn_css_url = ''; // Empty string = use local files
```

---

## 📝 Example CDN URLs:

### Cloudflare R2:
```php
$cdn_css_url = 'https://pub-abc123.r2.dev/css';
$cdn_js_url = 'https://pub-abc123.r2.dev/js';
```

### Amazon S3:
```php
$cdn_css_url = 'https://mybucket.s3.amazonaws.com/css';
$cdn_js_url = 'https://mybucket.s3.amazonaws.com/js';
```

### Custom CDN:
```php
$cdn_css_url = 'https://cdn.yourdomain.com/assets/css';
$cdn_js_url = 'https://cdn.yourdomain.com/assets/js';
```

---

## 🛠️ Troubleshooting:

### CSS Not Loading?
1. Check CDN URL is correct
2. Verify files are uploaded to CDN
3. Check file permissions (public read access)
4. Clear browser cache
5. Check browser console for 404 errors

### Want to Add More CSS/JS?
1. Upload file to CDN
2. Edit the appropriate include file (`header-css.php`, `footer-css.php`, or `footer-js.php`)
3. Add the link using the helper function:
```php
<link rel="stylesheet" href="<?php echo getCssUrl('your-new-file.css'); ?>">
```

---

## 📊 File Structure:

```
views/
├── includes/
│   ├── header-css.php    ← All header CSS links
│   ├── footer-css.php    ← All footer CSS links
│   └── footer-js.php     ← All footer JS links
├── header.view.php       ← Includes header-css.php
└── ...

sections/
└── views/
    └── footer.view.php   ← Includes footer-css.php & footer-js.php
```

---

## 🎯 Quick Start:

1. **Upload files to CDN**
2. **Edit 3 files:**
   - `views/includes/header-css.php` (line 8)
   - `views/includes/footer-css.php` (line 8)
   - `views/includes/footer-js.php` (line 8)
3. **Set CDN URL:**
   ```php
   $cdn_css_url = 'https://your-cdn-url.com/css';
   $cdn_js_url = 'https://your-cdn-url.com/js';
   ```
4. **Done!** All pages now load from CDN

---

## 📌 Important Notes:

- ✅ All individual CSS links have been removed from section files
- ✅ Everything is now centralized in 3 include files
- ✅ Helper functions automatically switch between CDN and local
- ✅ Cache busting is preserved with `?v=<?php echo time(); ?>`
- ✅ Backward compatible - leave CDN URL empty to use local files

---

**That's it!** Simple, centralized CDN management for all your redesign assets.
