# Breadcrumb Schema Implementation Guide
## Complete Setup for All Pages

This guide shows how to add breadcrumb schema to boost sitelinks on:
- ✅ Blog Posts
- ✅ Coupons
- ✅ Stores
- ✅ Categories
- ✅ Homepage

---

## Step 1: Add to Header (All Pages)

Add this to your main header file or each page's head section:

```php
<?php
require_once 'includes/breadcrumb_schema.php';
require_once 'includes/advanced_seo_system.php';
?>

<!-- Breadcrumb CSS -->
<link rel="stylesheet" href="<?php echo SITE_URL; ?>/assets/css/breadcrumb.css">
```

---

## Step 2: Implementation by Page Type

### 📝 Blog Post Page (single-blog.php)

**Already added!** The code is in place. Just add the HTML breadcrumb in the view:

```php
<!-- In views/single-blog.view.php, add after opening body tag -->
<div class="container">
    <?php 
    // Generate and display breadcrumb
    echo $breadcrumb->generateHTML();
    echo $breadcrumb->generateSchema();
    ?>
</div>
```

---

### 🎫 Coupon Page (single-coupon.php)

Add at the top after `require "core.php"`:

```php
<?php
require "core.php";
require "includes/breadcrumb_schema.php";
require "includes/advanced_seo_system.php";

// ... your existing code to fetch coupon ...

// After fetching coupon data, add:
$breadcrumb = BreadcrumbSchema::forCoupon(
    $coupon['coupon_title'],
    $coupon['coupon_slug'],
    $coupon['store_name'] ?? '',
    $coupon['store_slug'] ?? ''
);

$seo = new AdvancedSEOSystem($connect, 'https://alonedeals.com');
?>
```

Then in the view file, add:

```php
<!-- In views/single-coupon.view.php -->
<div class="container">
    <?php 
    echo $breadcrumb->generateHTML();
    echo $breadcrumb->generateSchema();
    ?>
</div>
```

---

### 🏪 Store Page (single-store.php)

Add at the top:

```php
<?php
require "core.php";
require "includes/breadcrumb_schema.php";
require "includes/advanced_seo_system.php";

// ... your existing code to fetch store ...

// After fetching store data:
$breadcrumb = BreadcrumbSchema::forStore(
    $store['store_title'],
    $store['store_slug']
);

$seo = new AdvancedSEOSystem($connect, 'https://alonedeals.com');
?>
```

In the view:

```php
<!-- In views/single-store.view.php -->
<div class="container">
    <?php 
    echo $breadcrumb->generateHTML();
    echo $breadcrumb->generateSchema();
    ?>
</div>
```

---

### 📂 Category Page (single-category.php)

Add at the top:

```php
<?php
require "core.php";
require "includes/breadcrumb_schema.php";
require "includes/advanced_seo_system.php";

// ... your existing code to fetch category ...

// After fetching category data:
$breadcrumb = BreadcrumbSchema::forCategory(
    $category['category_title'],
    $category['category_slug'],
    $category['parent_name'] ?? '',
    $category['parent_slug'] ?? ''
);

$seo = new AdvancedSEOSystem($connect, 'https://alonedeals.com');
?>
```

In the view:

```php
<!-- In views/single-category.view.php -->
<div class="container">
    <?php 
    echo $breadcrumb->generateHTML();
    echo $breadcrumb->generateSchema();
    ?>
</div>
```

---

### 🏠 Homepage (index.php)

Add at the top:

```php
<?php
require "core.php";
require "includes/breadcrumb_schema.php";
require "includes/advanced_seo_system.php";

// Homepage breadcrumb (just home)
$breadcrumb = new BreadcrumbSchema();

$seo = new AdvancedSEOSystem($connect, 'https://alonedeals.com');

// Generate Organization Schema for homepage
$orgSchema = $seo->generateStructuredData('organization', [
    'name' => 'AloneDeals',
    'logo' => 'logo.png',
    'email' => 'support@alonedeals.com',
    'social_links' => [
        'https://facebook.com/alonedeals',
        'https://twitter.com/alonedeals',
        'https://instagram.com/alonedeals'
    ]
]);

// Generate Website Schema
$websiteSchema = $seo->generateStructuredData('website', [
    'name' => 'AloneDeals - Best Coupons & Deals'
]);
?>
```

In the view:

```php
<!-- In views/home.view.php or index view -->
<?php 
echo $breadcrumb->generateSchema();
echo $orgSchema;
echo $websiteSchema;
?>
```

---

## Step 3: Quick Implementation Script

I'll create files that do this automatically for each page type.

---

## Files to Modify

1. **single-blog.php** ✅ (Already done)
2. **single-coupon.php** (Add breadcrumb)
3. **single-store.php** (Add breadcrumb)
4. **single-category.php** (Add breadcrumb)
5. **single-subcategory.php** (Add breadcrumb)
6. **index.php** (Add organization schema)
7. **blog.php** (Blog listing page)

---

## Expected Results

### Google Search Console
After implementation, you'll see:
- ✅ Breadcrumb rich results in testing tool
- ✅ Enhanced sitelinks in search results
- ✅ Better click-through rates
- ✅ Improved site structure understanding

### Timeline
- **Week 1**: Schema detected by Google
- **Week 2-3**: Breadcrumbs appear in search results
- **Month 1**: Enhanced sitelinks showing
- **Month 2+**: Improved rankings and CTR

---

## Testing

### 1. Test Schema Markup
Go to: https://search.google.com/test/rich-results
Enter your page URL and verify breadcrumb schema is detected.

### 2. Check Implementation
View page source and look for:
```json
{
  "@context": "https://schema.org",
  "@type": "BreadcrumbList",
  "itemListElement": [...]
}
```

### 3. Verify HTML
Check that breadcrumb navigation appears on page with proper microdata.

---

## Next Steps

I'll now create the actual implementation files for each page type.
