<?php
/**
 * Breadcrumb Schema Generator for Sitelinks
 * Automatically generates breadcrumb navigation and schema markup
 * Production URL: https://alonedeals.com
 */

class BreadcrumbSchema {
    private $siteUrl = 'https://alonedeals.com';
    private $siteName = 'AloneDeals';
    private $items = [];
    
    public function __construct() {
        // Always start with home
        $this->addItem('Home', $this->siteUrl);
    }
    
    /**
     * Add breadcrumb item
     */
    public function addItem($name, $url) {
        $this->items[] = [
            'name' => $name,
            'url' => $url
        ];
        return $this;
    }
    
    /**
     * Generate breadcrumb schema (JSON-LD)
     */
    public function generateSchema() {
        $itemList = [];
        
        foreach ($this->items as $index => $item) {
            $itemList[] = [
                "@type" => "ListItem",
                "position" => $index + 1,
                "name" => $item['name'],
                "item" => $item['url']
            ];
        }
        
        $schema = [
            "@context" => "https://schema.org",
            "@type" => "BreadcrumbList",
            "itemListElement" => $itemList
        ];
        
        return '<script type="application/ld+json">' . json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT) . '</script>';
    }
    
    /**
     * Generate HTML breadcrumb navigation
     */
    public function generateHTML($showHome = true) {
        if (count($this->items) <= 1 && !$showHome) {
            return '';
        }
        
        $html = '<nav aria-label="breadcrumb" class="breadcrumb-nav">';
        $html .= '<ol class="breadcrumb" itemscope itemtype="https://schema.org/BreadcrumbList">';
        
        $totalItems = count($this->items);
        foreach ($this->items as $index => $item) {
            $position = $index + 1;
            $isLast = ($position === $totalItems);
            
            $html .= '<li class="breadcrumb-item' . ($isLast ? ' active' : '') . '" itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem">';
            
            if ($isLast) {
                $html .= '<span itemprop="name">' . htmlspecialchars($item['name']) . '</span>';
            } else {
                $html .= '<a href="' . htmlspecialchars($item['url']) . '" itemprop="item">';
                $html .= '<span itemprop="name">' . htmlspecialchars($item['name']) . '</span>';
                $html .= '</a>';
            }
            
            $html .= '<meta itemprop="position" content="' . $position . '">';
            $html .= '</li>';
        }
        
        $html .= '</ol>';
        $html .= '</nav>';
        
        return $html;
    }
    
    /**
     * Auto-generate breadcrumbs based on URL
     */
    public static function autoGenerate($currentUrl, $pageTitle = '') {
        $breadcrumb = new self();
        
        // Parse URL
        $path = parse_url($currentUrl, PHP_URL_PATH);
        $segments = array_filter(explode('/', $path));
        
        $baseUrl = 'https://alonedeals.com';
        $currentPath = '';
        
        foreach ($segments as $segment) {
            $currentPath .= '/' . $segment;
            
            // Skip common files
            if (in_array($segment, ['index.php', 'index.html'])) {
                continue;
            }
            
            // Generate readable name
            $name = ucwords(str_replace(['-', '_', '.php'], [' ', ' ', ''], $segment));
            
            // Special cases
            if ($segment === 'Post') $name = 'Blog';
            if ($segment === 'single-blog.php') $name = $pageTitle ?: 'Blog Post';
            if ($segment === 'single-store.php') $name = $pageTitle ?: 'Store';
            if ($segment === 'single-coupon.php') $name = $pageTitle ?: 'Coupon';
            if ($segment === 'single-category.php') $name = $pageTitle ?: 'Category';
            
            $breadcrumb->addItem($name, $baseUrl . $currentPath);
        }
        
        // If page title provided and different from last item, update it
        if ($pageTitle && count($breadcrumb->items) > 0) {
            $lastIndex = count($breadcrumb->items) - 1;
            $breadcrumb->items[$lastIndex]['name'] = $pageTitle;
        }
        
        return $breadcrumb;
    }
    
    /**
     * Generate for blog post
     */
    public static function forBlogPost($postTitle, $postSlug, $categoryName = '', $categorySlug = '') {
        $breadcrumb = new self();
        $breadcrumb->addItem('Blog', 'https://alonedeals.com/Post');
        
        if ($categoryName && $categorySlug) {
            $breadcrumb->addItem($categoryName, 'https://alonedeals.com/Post/category/' . $categorySlug);
        }
        
        $breadcrumb->addItem($postTitle, 'https://alonedeals.com/Post/' . $postSlug);
        
        return $breadcrumb;
    }
    
    /**
     * Generate for store page
     */
    public static function forStore($storeName, $storeSlug) {
        $breadcrumb = new self();
        $breadcrumb->addItem('Stores', 'https://alonedeals.com/stores');
        $breadcrumb->addItem($storeName, 'https://alonedeals.com/store/' . $storeSlug);
        
        return $breadcrumb;
    }
    
    /**
     * Generate for coupon page
     */
    public static function forCoupon($couponTitle, $couponSlug, $storeName = '', $storeSlug = '') {
        $breadcrumb = new self();
        $breadcrumb->addItem('Coupons', 'https://alonedeals.com/coupons');
        
        if ($storeName && $storeSlug) {
            $breadcrumb->addItem($storeName, 'https://alonedeals.com/store/' . $storeSlug);
        }
        
        $breadcrumb->addItem($couponTitle, 'https://alonedeals.com/coupon/' . $couponSlug);
        
        return $breadcrumb;
    }
    
    /**
     * Generate for category page
     */
    public static function forCategory($categoryName, $categorySlug, $parentName = '', $parentSlug = '') {
        $breadcrumb = new self();
        $breadcrumb->addItem('Categories', 'https://alonedeals.com/categories');
        
        if ($parentName && $parentSlug) {
            $breadcrumb->addItem($parentName, 'https://alonedeals.com/category/' . $parentSlug);
        }
        
        $breadcrumb->addItem($categoryName, 'https://alonedeals.com/category/' . $categorySlug);
        
        return $breadcrumb;
    }
    
    /**
     * Generate for author page
     */
    public static function forAuthor($authorName, $authorSlug) {
        $breadcrumb = new self();
        $breadcrumb->addItem('Blog', 'https://alonedeals.com/Post');
        $breadcrumb->addItem('Authors', 'https://alonedeals.com/authors');
        $breadcrumb->addItem($authorName, 'https://alonedeals.com/author/' . $authorSlug);
        
        return $breadcrumb;
    }
}

/**
 * Helper function to quickly generate breadcrumbs
 */
function generate_breadcrumbs($type, $data) {
    switch($type) {
        case 'blog':
            return BreadcrumbSchema::forBlogPost(
                $data['title'],
                $data['slug'],
                $data['category_name'] ?? '',
                $data['category_slug'] ?? ''
            );
            
        case 'store':
            return BreadcrumbSchema::forStore($data['name'], $data['slug']);
            
        case 'coupon':
            return BreadcrumbSchema::forCoupon(
                $data['title'],
                $data['slug'],
                $data['store_name'] ?? '',
                $data['store_slug'] ?? ''
            );
            
        case 'category':
            return BreadcrumbSchema::forCategory(
                $data['name'],
                $data['slug'],
                $data['parent_name'] ?? '',
                $data['parent_slug'] ?? ''
            );
            
        case 'author':
            return BreadcrumbSchema::forAuthor($data['name'], $data['slug']);
            
        default:
            return BreadcrumbSchema::autoGenerate($_SERVER['REQUEST_URI'], $data['title'] ?? '');
    }
}
?>
