# Enable PHP Error Display

## Option 1: Add to Top of index.php (Recommended)

Add this code at the **very top** of `index.php` (before any other code):

```php
<?php
// Enable error display for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);

// Rest of your code...
require "core.php";
// ...
?>
```

## Option 2: Add to core.php

Add this at the **very top** of `core.php`:

```php
<?php
// Enable error display
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);

// Rest of core.php code...
?>
```

## Option 3: Create .htaccess File

Add this to your `.htaccess` file in the root directory:

```apache
# Enable PHP error display
php_flag display_errors on
php_value error_reporting E_ALL
php_flag display_startup_errors on
```

## Option 4: Edit php.ini (If you have access)

Find and edit these lines in `php.ini`:

```ini
display_errors = On
display_startup_errors = On
error_reporting = E_ALL
log_errors = On
error_log = /path/to/php_errors.log
```

Then restart Apache/PHP-FPM.

## Option 5: Use the show_errors.php File

I created a file called `show_errors.php`. Include it at the top of any file:

```php
<?php
require_once 'show_errors.php';

// Your code here...
?>
```

## Quick Test

Create a test file `test_errors.php`:

```php
<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "Error display is enabled!<br>";

// This will cause an error to test
echo $undefined_variable;

// This will also cause an error
trigger_error("This is a test error", E_USER_WARNING);
?>
```

Access: `https://www.alonedeals.com/test_errors.php`

You should see error messages displayed.

## For Production (Disable Errors)

When you're done debugging, **disable error display** for security:

```php
<?php
error_reporting(0);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', '/path/to/error.log');
?>
```

This will hide errors from visitors but still log them to a file.

## Check Current Error Settings

Create `check_php_config.php`:

```php
<?php
phpinfo();
?>
```

Access it and search for:
- `display_errors`
- `error_reporting`
- `log_errors`

## Common PHP Errors to Look For

1. **Undefined variable**: Using a variable that doesn't exist
2. **Undefined function**: Calling a function that doesn't exist
3. **Parse error**: Syntax error in PHP code
4. **Fatal error**: Critical error that stops execution
5. **Warning**: Non-critical error
6. **Notice**: Minor issue (like undefined variable)

## Debugging the CSS Issue

To debug why CSS is loading from CDN, add this to `index.php`:

```php
<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

require "core.php";

// Debug: Check what getCssUrl returns
include 'views/includes/header-css.php';
echo "<!-- DEBUG: getCssUrl('test.css') = " . getCssUrl('test.css') . " -->";
echo "<!-- DEBUG: cdn_css_url = '" . $cdn_css_url . "' -->";

// Rest of code...
?>
```

Then view page source and look for the DEBUG comments.
