# Force Local CSS and Font Loading - Complete Guide

## Problem
CSS files are loading from CDN (assets.alonedeals.com) even though `$cdn_css_url = ''` is empty.

## Root Causes
1. **Browser Cache** - Old CDN URLs cached in browser
2. **Server Cache** - PHP OPcache or APCu caching old code
3. **CDN/Proxy Cache** - Cloudflare or similar service caching
4. **Service Worker** - PWA service worker caching old files

## Solution Steps (Do ALL of these)

### Step 1: Clear Server Cache
Run this file in your browser:
```
https://www.alonedeals.com/clear_server_cache.php
```

### Step 2: Clear Browser Cache (IMPORTANT!)

**Chrome:**
1. Press `Ctrl+Shift+Delete` (Windows) or `Cmd+Shift+Delete` (Mac)
2. Select "All time"
3. Check "Cached images and files"
4. Click "Clear data"

**Firefox:**
1. Press `Ctrl+Shift+Delete`
2. Select "Everything"
3. Check "Cache"
4. Click "Clear Now"

**Or use Incognito/Private Mode:**
- Chrome: `Ctrl+Shift+N`
- Firefox: `Ctrl+Shift+P`

### Step 3: Hard Refresh
After clearing cache, do a hard refresh:
- Windows: `Ctrl+F5` or `Ctrl+Shift+R`
- Mac: `Cmd+Shift+R`

### Step 4: Clear Cloudflare Cache (if using)
1. Log into Cloudflare dashboard
2. Go to your domain
3. Click "Caching" → "Configuration"
4. Click "Purge Everything"

### Step 5: Disable Service Worker (if exists)

**Chrome DevTools:**
1. Press F12
2. Go to "Application" tab
3. Click "Service Workers" in left sidebar
4. Click "Unregister" for any service workers
5. Reload page

### Step 6: Check What's Loading

**Open DevTools (F12) → Network Tab:**

1. Reload page
2. Filter by "CSS"
3. Check each CSS file URL

**Should see:**
```
✅ https://www.alonedeals.com/assets/css/sidenav-redesign.css?v=1234567890
✅ https://www.alonedeals.com/assets/css/tabler-icons.min.css?v=1234567890
```

**Should NOT see:**
```
❌ https://assets.alonedeals.com/assets/css/...
```

4. Filter by "Font"
5. Check font file URLs

**Should see:**
```
✅ https://www.alonedeals.com/assets/fonts/tabler-icons.woff2
✅ https://www.alonedeals.com/assets/fonts/tabler-icons.woff
```

## What We Changed

### 1. Added Cache Busting
All CSS files now have `?v=<?php echo time(); ?>` parameter:
```php
<link rel="stylesheet" href="<?php echo getCssUrl('sidenav-redesign.css'); ?>?v=<?php echo time(); ?>">
```

This forces browser to reload files on every page load.

### 2. Confirmed CDN is Disabled
```php
$cdn_css_url = ''; // DISABLED - Using local CSS and fonts
```

### 3. Font CSS Has Cache Buster
```php
<link rel="stylesheet" href="<?php echo $urlPath->assets_css('tabler-icons.min.css'); ?>?v=<?php echo time(); ?>">
```

## Verify Configuration

### Check views/includes/header-css.php
Line 10 should be:
```php
$cdn_css_url = ''; // DISABLED - Using local CSS and fonts
```

### Check views/includes/footer-js.php
Line 8 should be:
```php
$cdn_js_url = ''; // DISABLED - Using local JS
```

Or if you want to use CDN for JS but not CSS:
```php
$cdn_js_url = 'https://assets.alonedeals.com/assets/js'; // CDN for JS only
```

## Still Not Working?

### Test Direct File Access
Try accessing files directly in browser:

1. **CSS File:**
   ```
   https://www.alonedeals.com/assets/css/tabler-icons.min.css
   ```
   Should load successfully (not 404)

2. **Font Files:**
   ```
   https://www.alonedeals.com/assets/fonts/tabler-icons.woff2
   https://www.alonedeals.com/assets/fonts/tabler-icons.woff
   https://www.alonedeals.com/assets/fonts/tabler-icons.ttf
   ```
   Should download or display (not 404)

If any return 404, check:
- File permissions: `chmod 644 assets/fonts/*`
- File exists: `ls -la assets/fonts/`
- .htaccess not blocking: Check for `Deny from all` rules

### Check .htaccess
Make sure your `.htaccess` doesn't block font files:
```apache
# Allow font files
<FilesMatch "\.(ttf|otf|eot|woff|woff2)$">
    <IfModule mod_headers.c>
        Header set Access-Control-Allow-Origin "*"
    </IfModule>
</FilesMatch>
```

### Check PHP Configuration
Create a test file `test_cdn.php`:
```php
<?php
require 'core.php';
echo "CDN CSS URL: '" . (isset($cdn_css_url) ? $cdn_css_url : 'not set') . "'<br>";
echo "Should be empty!<br>";
echo "<br>CSS URL function test:<br>";
include 'views/includes/header-css.php';
echo getCssUrl('test.css');
?>
```

Access: `https://www.alonedeals.com/test_cdn.php`

Should output:
```
CDN CSS URL: ''
Should be empty!
CSS URL function test:
/assets/css/test.css
```

## Final Checklist

- [ ] Cleared server cache (ran clear_server_cache.php)
- [ ] Cleared browser cache completely
- [ ] Did hard refresh (Ctrl+F5)
- [ ] Tested in Incognito/Private mode
- [ ] Cleared Cloudflare cache (if applicable)
- [ ] Unregistered service workers
- [ ] Verified CSS loads from www.alonedeals.com
- [ ] Verified fonts load from www.alonedeals.com
- [ ] No CORS errors in console

## Success Indicators

✅ No CORS errors in browser console
✅ All CSS files load from `www.alonedeals.com`
✅ All font files load from `www.alonedeals.com`
✅ Page displays correctly with all styles
✅ Icons display correctly

## If Still Seeing CDN URLs

The cache is VERY persistent. Try:

1. **Different Browser** - Test in a browser you haven't used
2. **Different Device** - Test on phone/tablet
3. **Different Network** - Use mobile data instead of WiFi
4. **Wait 5 Minutes** - Some caches have TTL (Time To Live)

If it works in a fresh browser/device, it's definitely a cache issue on your main browser.
