<?php
session_start();

require '../../config.php';
require '../admin_config.php';
require '../functions.php';

if (isset($_SESSION['user_email'])){

$connect = connect($database);

if (!isAdmin($connect) && !isAgent($connect)) {
    header('Location: ../../index.php');
    exit;
}

if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['import_from_affiliate'])) {
    $program = cleardata($_POST['affiliate_program']);
    $import_count = 0;
    $error_messages = [];
    
    // Get program settings
    $stmt = $connect->prepare("SELECT * FROM affiliate_programs WHERE program_name = ? AND status = 1");
    $stmt->execute([$program]);
    $program_settings = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Check if program has API key configured
    if (empty($program_settings['api_key'])) {
        $error_messages = ["❌ " . $program . " API key is not configured. Please configure the API credentials first."];
    }
    
    if (!$program_settings) {
        $error_messages[] = "Program not found or not enabled.";
    } else {
        switch ($program) {
            case 'cuelinks':
                $result = importFromCuelinks($connect, $program_settings);
                $import_count = $result['count'];
                $error_messages = array_merge($error_messages, $result['errors']);
                break;
                
            case 'inrdeals':
                $result = importFromINRDeals($connect, $program_settings);
                $import_count = $result['count'];
                $error_messages = array_merge($error_messages, $result['errors']);
                break;
                
            case 'admitad':
                $result = importFromAdmitad($connect, $program_settings);
                $import_count = $result['count'];
                $error_messages = array_merge($error_messages, $result['errors']);
                break;
                
            case 'cj':
                $result = importFromCJ($connect, $program_settings);
                $import_count = $result['count'];
                $error_messages = array_merge($error_messages, $result['errors']);
                break;
        }
    }
}

function importFromCuelinks($connect, $settings) {
    $count = 0;
    $errors = [];
    
    try {
        // Cuelinks API endpoint for deals/coupons
        $url = "https://api.cuelinks.com/v1/deals";
        $headers = [
            'Authorization: Bearer ' . $settings['api_key'],
            'Content-Type: application/json',
            'Accept: application/json'
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code == 200) {
            $data = json_decode($response, true);
            if (isset($data['data']) && is_array($data['data'])) {
                foreach ($data['data'] as $deal) {
                    if (importCouponFromData($connect, $deal, 'cuelinks')) {
                        $count++;
                    }
                }
            } elseif (isset($data['deals']) && is_array($data['deals'])) {
                foreach ($data['deals'] as $deal) {
                    if (importCouponFromData($connect, $deal, 'cuelinks')) {
                        $count++;
                    }
                }
            }
        } else {
            $errors[] = "Cuelinks API error: HTTP $http_code - " . $response;
        }
    } catch (Exception $e) {
        $errors[] = "Cuelinks import error: " . $e->getMessage();
    }
    
    return ['count' => $count, 'errors' => $errors];
}

function importFromINRDeals($connect, $settings) {
    $count = 0;
    $errors = [];
    
    try {
        // INR Deals API endpoint
        $url = "https://api.inrdeals.com/v1/coupons";
        $headers = [
            'Authorization: Bearer ' . $settings['api_key'],
            'Content-Type: application/json',
            'Accept: application/json'
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code == 200) {
            $data = json_decode($response, true);
            if (isset($data['data']) && is_array($data['data'])) {
                foreach ($data['data'] as $coupon) {
                    if (importCouponFromData($connect, $coupon, 'inrdeals')) {
                        $count++;
                    }
                }
            } elseif (isset($data['coupons']) && is_array($data['coupons'])) {
                foreach ($data['coupons'] as $coupon) {
                    if (importCouponFromData($connect, $coupon, 'inrdeals')) {
                        $count++;
                    }
                }
            }
        } else {
            $errors[] = "INR Deals API error: HTTP $http_code - " . $response;
        }
    } catch (Exception $e) {
        $errors[] = "INR Deals import error: " . $e->getMessage();
    }
    
    return ['count' => $count, 'errors' => $errors];
}

function importFromAdmitad($connect, $settings) {
    $count = 0;
    $errors = [];
    
    try {
        // First get access token using Base64 authentication
        $token_url = "https://api.admitad.com/token/";
        
        // Create Base64 encoded credentials
        $credentials = base64_encode($settings['api_key'] . ':' . $settings['api_secret']);
        
        $token_data = [
            'grant_type' => 'client_credentials',
            'scope' => 'coupons'
        ];
        
        $headers = [
            'Authorization: Basic ' . $credentials,
            'Content-Type: application/x-www-form-urlencoded'
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $token_url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($token_data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $token_response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code == 200) {
            $token_data = json_decode($token_response, true);
            if (isset($token_data['access_token'])) {
                $access_token = $token_data['access_token'];
                
                // Now get coupons using proper Admitad API endpoint
                $coupons_url = "https://api.admitad.com/coupons/";
                $headers = [
                    'Authorization: Bearer ' . $access_token,
                    'Content-Type: application/json'
                ];
                
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, $coupons_url);
                curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                curl_setopt($ch, CURLOPT_TIMEOUT, 30);
                
                $response = curl_exec($ch);
                $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                curl_close($ch);
                
                if ($http_code == 200) {
                    $data = json_decode($response, true);
                    if (isset($data['results']) && is_array($data['results'])) {
                        foreach ($data['results'] as $coupon) {
                            if (importCouponFromData($connect, $coupon, 'admitad')) {
                                $count++;
                            }
                        }
                    }
                } else {
                    $errors[] = "Admitad coupons API error: HTTP $http_code";
                }
            } else {
                $errors[] = "Failed to get Admitad access token";
            }
        } else {
            $errors[] = "Admitad token API error: HTTP $http_code";
        }
    } catch (Exception $e) {
        $errors[] = "Admitad import error: " . $e->getMessage();
    }
    
    return ['count' => $count, 'errors' => $errors];
}

function importFromCJ($connect, $settings) {
    $count = 0;
    $errors = [];
    
    try {
        // Commission Junction API endpoint for product search
        $url = "https://product-search.api.cj.com/v3/product-search";
        $headers = [
            'Authorization: Bearer ' . $settings['api_key'],
            'Content-Type: application/json',
            'Accept: application/json'
        ];
        
        // Add query parameters
        $params = [
            'website-id' => $settings['api_secret'], // Using api_secret as publisher_id
            'format' => 'json',
            'page-number' => 1,
            'records-per-page' => 100
        ];
        
        $url .= '?' . http_build_query($params);
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code == 200) {
            $data = json_decode($response, true);
            if (isset($data['products']) && is_array($data['products'])) {
                foreach ($data['products'] as $product) {
                    if (importCouponFromData($connect, $product, 'cj')) {
                        $count++;
                    }
                }
            } elseif (isset($data['links']) && is_array($data['links'])) {
                foreach ($data['links'] as $link) {
                    if (importCouponFromData($connect, $link, 'cj')) {
                        $count++;
                    }
                }
            }
        } else {
            $errors[] = "CJ API error: HTTP $http_code - " . $response;
        }
    } catch (Exception $e) {
        $errors[] = "CJ import error: " . $e->getMessage();
    }
    
    return ['count' => $count, 'errors' => $errors];
}

function importCouponFromData($connect, $data, $source) {
    try {
        // Extract common fields based on source
        $title = '';
        $description = '';
        $code = '';
        $link = '';
        $image_url = '';
        $store_name = '';
        $expiry = null;
        
        switch ($source) {
            case 'cuelinks':
                $title = $data['title'] ?? $data['deal_title'] ?? '';
                $description = $data['description'] ?? $data['deal_description'] ?? '';
                $code = $data['coupon_code'] ?? $data['code'] ?? '';
                $link = $data['affiliate_url'] ?? $data['url'] ?? '';
                $image_url = $data['image_url'] ?? $data['image'] ?? '';
                $store_name = $data['store_name'] ?? $data['merchant_name'] ?? '';
                $expiry = $data['expiry_date'] ?? $data['expires_at'] ?? null;
                break;
                
            case 'inrdeals':
                $title = $data['title'] ?? $data['coupon_title'] ?? '';
                $description = $data['description'] ?? $data['coupon_description'] ?? '';
                $code = $data['code'] ?? $data['coupon_code'] ?? '';
                $link = $data['url'] ?? $data['affiliate_url'] ?? '';
                $image_url = $data['image'] ?? $data['image_url'] ?? '';
                $store_name = $data['store'] ?? $data['merchant'] ?? '';
                $expiry = $data['expires'] ?? $data['expiry_date'] ?? null;
                break;
                
            case 'admitad':
                $title = $data['name'] ?? $data['title'] ?? '';
                $description = $data['description'] ?? $data['short_description'] ?? '';
                $code = $data['promocode'] ?? $data['code'] ?? '';
                $link = $data['goto_link'] ?? $data['url'] ?? '';
                $image_url = $data['image'] ?? $data['image_url'] ?? '';
                $store_name = $data['campaign']['name'] ?? $data['merchant_name'] ?? '';
                $expiry = $data['date_end'] ?? $data['expires_at'] ?? null;
                break;
                
            case 'cj':
                $title = $data['link-name'] ?? $data['name'] ?? '';
                $description = $data['description'] ?? $data['short-description'] ?? '';
                $code = $data['promo-code'] ?? $data['code'] ?? '';
                $link = $data['clickUrl'] ?? $data['url'] ?? '';
                $image_url = $data['image-url'] ?? $data['image'] ?? '';
                $store_name = $data['advertiser-name'] ?? $data['merchant-name'] ?? '';
                $expiry = $data['expiry-date'] ?? $data['expires-at'] ?? null;
                break;
        }
        
        if (empty($title) || empty($link)) {
            return false;
        }
        
        // Check if coupon already exists
        $stmt = $connect->prepare("SELECT coupon_id FROM coupons WHERE coupon_title = ? AND coupon_link = ?");
        $stmt->execute([$title, $link]);
        if ($stmt->fetch()) {
            return false; // Already exists
        }
        
        // Get or create store
        $store_id = getOrCreateStore($connect, $store_name, $link, $image_url);
        
        // Download and save image
        $image_filename = '';
        if (!empty($image_url)) {
            $image_filename = downloadAndSaveImage($image_url, $title);
        }
        
        // Insert coupon
        $stmt = $connect->prepare("INSERT INTO coupons (coupon_id, coupon_title, coupon_tagline, coupon_description, coupon_code, coupon_link, coupon_image, store_id, coupon_featured, coupon_status, coupon_expire) VALUES (null, ?, ?, ?, ?, ?, ?, ?, 0, 1, ?)");
        
        return $stmt->execute([
            $title,
            $description,
            $description,
            $code,
            $link,
            $image_filename,
            $store_id,
            $expiry
        ]);
        
    } catch (Exception $e) {
        return false;
    }
}

function getOrCreateStore($connect, $store_name, $store_url, $image_url) {
    if (empty($store_name)) {
        return null;
    }
    
    // Check if store exists
    $stmt = $connect->prepare("SELECT store_id FROM stores WHERE store_title = ?");
    $stmt->execute([$store_name]);
    $store = $stmt->fetch();
    
    if ($store) {
        return $store['store_id'];
    }
    
    // Create new store
    $slug = strtolower(str_replace(' ', '-', $store_name));
    $slug = preg_replace('/[^a-z0-9\-]/', '', $slug);
    
    // Download store image
    $store_image = '';
    if (!empty($image_url)) {
        $store_image = downloadAndSaveImage($image_url, $store_name . '_store');
    }
    
    $stmt = $connect->prepare("INSERT INTO stores (store_id, store_title, store_slug, store_description, store_seotitle, store_seodescription, store_featured, store_status, store_image, store_url) VALUES (null, ?, ?, '', ?, '', 0, 1, ?, ?)");
    
    if ($stmt->execute([$store_name, $slug, $store_name, $store_image, $store_url])) {
        return $connect->lastInsertId();
    }
    
    return null;
}

function downloadAndSaveImage($url, $filename) {
    try {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        
        $image_data = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code == 200 && $image_data) {
            $extension = pathinfo(parse_url($url, PHP_URL_PATH), PATHINFO_EXTENSION);
            if (empty($extension)) {
                $extension = 'jpg';
            }
            
            $filename = preg_replace('/[^a-zA-Z0-9\-_]/', '', $filename);
            $filename = $filename . '_' . time() . '.' . $extension;
            $filepath = '../../images/' . $filename;
            
            if (file_put_contents($filepath, $image_data)) {
                return $filename;
            }
        }
    } catch (Exception $e) {
        // Ignore image download errors
    }
    
    return '';
}

// Get available programs (all enabled programs, regardless of API key status)
$stmt = $connect->prepare("SELECT * FROM affiliate_programs WHERE status = 1");
$stmt->execute();
$available_programs = $stmt->fetchAll(PDO::FETCH_ASSOC);

require '../views/header.view.php';
require '../views/affiliate_import.view.php';
require '../views/footer.view.php';

} else {
    header('Location: ./login.php');
}
?>


require '../../config.php';
require '../admin_config.php';
require '../functions.php';

if (isset($_SESSION['user_email'])){

$connect = connect($database);

if (!isAdmin($connect) && !isAgent($connect)) {
    header('Location: ../../index.php');
    exit;
}

if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['import_from_affiliate'])) {
    $program = cleardata($_POST['affiliate_program']);
    $import_count = 0;
    $error_messages = [];
    
    // Get program settings
    $stmt = $connect->prepare("SELECT * FROM affiliate_programs WHERE program_name = ? AND status = 1");
    $stmt->execute([$program]);
    $program_settings = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Check if program has API key configured
    if (empty($program_settings['api_key'])) {
        $error_messages = ["❌ " . $program . " API key is not configured. Please configure the API credentials first."];
    }
    
    if (!$program_settings) {
        $error_messages[] = "Program not found or not enabled.";
    } else {
        switch ($program) {
            case 'cuelinks':
                $result = importFromCuelinks($connect, $program_settings);
                $import_count = $result['count'];
                $error_messages = array_merge($error_messages, $result['errors']);
                break;
                
            case 'inrdeals':
                $result = importFromINRDeals($connect, $program_settings);
                $import_count = $result['count'];
                $error_messages = array_merge($error_messages, $result['errors']);
                break;
                
            case 'admitad':
                $result = importFromAdmitad($connect, $program_settings);
                $import_count = $result['count'];
                $error_messages = array_merge($error_messages, $result['errors']);
                break;
                
            case 'cj':
                $result = importFromCJ($connect, $program_settings);
                $import_count = $result['count'];
                $error_messages = array_merge($error_messages, $result['errors']);
                break;
        }
    }
}

function importFromCuelinks($connect, $settings) {
    $count = 0;
    $errors = [];
    
    try {
        // Cuelinks API endpoint for deals/coupons
        $url = "https://api.cuelinks.com/v1/deals";
        $headers = [
            'Authorization: Bearer ' . $settings['api_key'],
            'Content-Type: application/json',
            'Accept: application/json'
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code == 200) {
            $data = json_decode($response, true);
            if (isset($data['data']) && is_array($data['data'])) {
                foreach ($data['data'] as $deal) {
                    if (importCouponFromData($connect, $deal, 'cuelinks')) {
                        $count++;
                    }
                }
            } elseif (isset($data['deals']) && is_array($data['deals'])) {
                foreach ($data['deals'] as $deal) {
                    if (importCouponFromData($connect, $deal, 'cuelinks')) {
                        $count++;
                    }
                }
            }
        } else {
            $errors[] = "Cuelinks API error: HTTP $http_code - " . $response;
        }
    } catch (Exception $e) {
        $errors[] = "Cuelinks import error: " . $e->getMessage();
    }
    
    return ['count' => $count, 'errors' => $errors];
}

function importFromINRDeals($connect, $settings) {
    $count = 0;
    $errors = [];
    
    try {
        // INR Deals API endpoint
        $url = "https://api.inrdeals.com/v1/coupons";
        $headers = [
            'Authorization: Bearer ' . $settings['api_key'],
            'Content-Type: application/json',
            'Accept: application/json'
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code == 200) {
            $data = json_decode($response, true);
            if (isset($data['data']) && is_array($data['data'])) {
                foreach ($data['data'] as $coupon) {
                    if (importCouponFromData($connect, $coupon, 'inrdeals')) {
                        $count++;
                    }
                }
            } elseif (isset($data['coupons']) && is_array($data['coupons'])) {
                foreach ($data['coupons'] as $coupon) {
                    if (importCouponFromData($connect, $coupon, 'inrdeals')) {
                        $count++;
                    }
                }
            }
        } else {
            $errors[] = "INR Deals API error: HTTP $http_code - " . $response;
        }
    } catch (Exception $e) {
        $errors[] = "INR Deals import error: " . $e->getMessage();
    }
    
    return ['count' => $count, 'errors' => $errors];
}

function importFromAdmitad($connect, $settings) {
    $count = 0;
    $errors = [];
    
    try {
        // First get access token using Base64 authentication
        $token_url = "https://api.admitad.com/token/";
        
        // Create Base64 encoded credentials
        $credentials = base64_encode($settings['api_key'] . ':' . $settings['api_secret']);
        
        $token_data = [
            'grant_type' => 'client_credentials',
            'scope' => 'coupons'
        ];
        
        $headers = [
            'Authorization: Basic ' . $credentials,
            'Content-Type: application/x-www-form-urlencoded'
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $token_url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($token_data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $token_response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code == 200) {
            $token_data = json_decode($token_response, true);
            if (isset($token_data['access_token'])) {
                $access_token = $token_data['access_token'];
                
                // Now get coupons using proper Admitad API endpoint
                $coupons_url = "https://api.admitad.com/coupons/";
                $headers = [
                    'Authorization: Bearer ' . $access_token,
                    'Content-Type: application/json'
                ];
                
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, $coupons_url);
                curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                curl_setopt($ch, CURLOPT_TIMEOUT, 30);
                
                $response = curl_exec($ch);
                $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                curl_close($ch);
                
                if ($http_code == 200) {
                    $data = json_decode($response, true);
                    if (isset($data['results']) && is_array($data['results'])) {
                        foreach ($data['results'] as $coupon) {
                            if (importCouponFromData($connect, $coupon, 'admitad')) {
                                $count++;
                            }
                        }
                    }
                } else {
                    $errors[] = "Admitad coupons API error: HTTP $http_code";
                }
            } else {
                $errors[] = "Failed to get Admitad access token";
            }
        } else {
            $errors[] = "Admitad token API error: HTTP $http_code";
        }
    } catch (Exception $e) {
        $errors[] = "Admitad import error: " . $e->getMessage();
    }
    
    return ['count' => $count, 'errors' => $errors];
}

function importFromCJ($connect, $settings) {
    $count = 0;
    $errors = [];
    
    try {
        // Commission Junction API endpoint for product search
        $url = "https://product-search.api.cj.com/v3/product-search";
        $headers = [
            'Authorization: Bearer ' . $settings['api_key'],
            'Content-Type: application/json',
            'Accept: application/json'
        ];
        
        // Add query parameters
        $params = [
            'website-id' => $settings['api_secret'], // Using api_secret as publisher_id
            'format' => 'json',
            'page-number' => 1,
            'records-per-page' => 100
        ];
        
        $url .= '?' . http_build_query($params);
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code == 200) {
            $data = json_decode($response, true);
            if (isset($data['products']) && is_array($data['products'])) {
                foreach ($data['products'] as $product) {
                    if (importCouponFromData($connect, $product, 'cj')) {
                        $count++;
                    }
                }
            } elseif (isset($data['links']) && is_array($data['links'])) {
                foreach ($data['links'] as $link) {
                    if (importCouponFromData($connect, $link, 'cj')) {
                        $count++;
                    }
                }
            }
        } else {
            $errors[] = "CJ API error: HTTP $http_code - " . $response;
        }
    } catch (Exception $e) {
        $errors[] = "CJ import error: " . $e->getMessage();
    }
    
    return ['count' => $count, 'errors' => $errors];
}

function importCouponFromData($connect, $data, $source) {
    try {
        // Extract common fields based on source
        $title = '';
        $description = '';
        $code = '';
        $link = '';
        $image_url = '';
        $store_name = '';
        $expiry = null;
        
        switch ($source) {
            case 'cuelinks':
                $title = $data['title'] ?? $data['deal_title'] ?? '';
                $description = $data['description'] ?? $data['deal_description'] ?? '';
                $code = $data['coupon_code'] ?? $data['code'] ?? '';
                $link = $data['affiliate_url'] ?? $data['url'] ?? '';
                $image_url = $data['image_url'] ?? $data['image'] ?? '';
                $store_name = $data['store_name'] ?? $data['merchant_name'] ?? '';
                $expiry = $data['expiry_date'] ?? $data['expires_at'] ?? null;
                break;
                
            case 'inrdeals':
                $title = $data['title'] ?? $data['coupon_title'] ?? '';
                $description = $data['description'] ?? $data['coupon_description'] ?? '';
                $code = $data['code'] ?? $data['coupon_code'] ?? '';
                $link = $data['url'] ?? $data['affiliate_url'] ?? '';
                $image_url = $data['image'] ?? $data['image_url'] ?? '';
                $store_name = $data['store'] ?? $data['merchant'] ?? '';
                $expiry = $data['expires'] ?? $data['expiry_date'] ?? null;
                break;
                
            case 'admitad':
                $title = $data['name'] ?? $data['title'] ?? '';
                $description = $data['description'] ?? $data['short_description'] ?? '';
                $code = $data['promocode'] ?? $data['code'] ?? '';
                $link = $data['goto_link'] ?? $data['url'] ?? '';
                $image_url = $data['image'] ?? $data['image_url'] ?? '';
                $store_name = $data['campaign']['name'] ?? $data['merchant_name'] ?? '';
                $expiry = $data['date_end'] ?? $data['expires_at'] ?? null;
                break;
                
            case 'cj':
                $title = $data['link-name'] ?? $data['name'] ?? '';
                $description = $data['description'] ?? $data['short-description'] ?? '';
                $code = $data['promo-code'] ?? $data['code'] ?? '';
                $link = $data['clickUrl'] ?? $data['url'] ?? '';
                $image_url = $data['image-url'] ?? $data['image'] ?? '';
                $store_name = $data['advertiser-name'] ?? $data['merchant-name'] ?? '';
                $expiry = $data['expiry-date'] ?? $data['expires-at'] ?? null;
                break;
        }
        
        if (empty($title) || empty($link)) {
            return false;
        }
        
        // Check if coupon already exists
        $stmt = $connect->prepare("SELECT coupon_id FROM coupons WHERE coupon_title = ? AND coupon_link = ?");
        $stmt->execute([$title, $link]);
        if ($stmt->fetch()) {
            return false; // Already exists
        }
        
        // Get or create store
        $store_id = getOrCreateStore($connect, $store_name, $link, $image_url);
        
        // Download and save image
        $image_filename = '';
        if (!empty($image_url)) {
            $image_filename = downloadAndSaveImage($image_url, $title);
        }
        
        // Insert coupon
        $stmt = $connect->prepare("INSERT INTO coupons (coupon_id, coupon_title, coupon_tagline, coupon_description, coupon_code, coupon_link, coupon_image, store_id, coupon_featured, coupon_status, coupon_expire) VALUES (null, ?, ?, ?, ?, ?, ?, ?, 0, 1, ?)");
        
        return $stmt->execute([
            $title,
            $description,
            $description,
            $code,
            $link,
            $image_filename,
            $store_id,
            $expiry
        ]);
        
    } catch (Exception $e) {
        return false;
    }
}

function getOrCreateStore($connect, $store_name, $store_url, $image_url) {
    if (empty($store_name)) {
        return null;
    }
    
    // Check if store exists
    $stmt = $connect->prepare("SELECT store_id FROM stores WHERE store_title = ?");
    $stmt->execute([$store_name]);
    $store = $stmt->fetch();
    
    if ($store) {
        return $store['store_id'];
    }
    
    // Create new store
    $slug = strtolower(str_replace(' ', '-', $store_name));
    $slug = preg_replace('/[^a-z0-9\-]/', '', $slug);
    
    // Download store image
    $store_image = '';
    if (!empty($image_url)) {
        $store_image = downloadAndSaveImage($image_url, $store_name . '_store');
    }
    
    $stmt = $connect->prepare("INSERT INTO stores (store_id, store_title, store_slug, store_description, store_seotitle, store_seodescription, store_featured, store_status, store_image, store_url) VALUES (null, ?, ?, '', ?, '', 0, 1, ?, ?)");
    
    if ($stmt->execute([$store_name, $slug, $store_name, $store_image, $store_url])) {
        return $connect->lastInsertId();
    }
    
    return null;
}

function downloadAndSaveImage($url, $filename) {
    try {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        
        $image_data = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code == 200 && $image_data) {
            $extension = pathinfo(parse_url($url, PHP_URL_PATH), PATHINFO_EXTENSION);
            if (empty($extension)) {
                $extension = 'jpg';
            }
            
            $filename = preg_replace('/[^a-zA-Z0-9\-_]/', '', $filename);
            $filename = $filename . '_' . time() . '.' . $extension;
            $filepath = '../../images/' . $filename;
            
            if (file_put_contents($filepath, $image_data)) {
                return $filename;
            }
        }
    } catch (Exception $e) {
        // Ignore image download errors
    }
    
    return '';
}

// Get available programs (all enabled programs, regardless of API key status)
$stmt = $connect->prepare("SELECT * FROM affiliate_programs WHERE status = 1");
$stmt->execute();
$available_programs = $stmt->fetchAll(PDO::FETCH_ASSOC);

require '../views/header.view.php';
require '../views/affiliate_import.view.php';
require '../views/footer.view.php';

} else {
    header('Location: ./login.php');
}
?>


require '../../config.php';
require '../admin_config.php';
require '../functions.php';

if (isset($_SESSION['user_email'])){

$connect = connect($database);

if (!isAdmin($connect) && !isAgent($connect)) {
    header('Location: ../../index.php');
    exit;
}

if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['import_from_affiliate'])) {
    $program = cleardata($_POST['affiliate_program']);
    $import_count = 0;
    $error_messages = [];
    
    // Get program settings
    $stmt = $connect->prepare("SELECT * FROM affiliate_programs WHERE program_name = ? AND status = 1");
    $stmt->execute([$program]);
    $program_settings = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Check if program has API key configured
    if (empty($program_settings['api_key'])) {
        $error_messages = ["❌ " . $program . " API key is not configured. Please configure the API credentials first."];
    }
    
    if (!$program_settings) {
        $error_messages[] = "Program not found or not enabled.";
    } else {
        switch ($program) {
            case 'cuelinks':
                $result = importFromCuelinks($connect, $program_settings);
                $import_count = $result['count'];
                $error_messages = array_merge($error_messages, $result['errors']);
                break;
                
            case 'inrdeals':
                $result = importFromINRDeals($connect, $program_settings);
                $import_count = $result['count'];
                $error_messages = array_merge($error_messages, $result['errors']);
                break;
                
            case 'admitad':
                $result = importFromAdmitad($connect, $program_settings);
                $import_count = $result['count'];
                $error_messages = array_merge($error_messages, $result['errors']);
                break;
                
            case 'cj':
                $result = importFromCJ($connect, $program_settings);
                $import_count = $result['count'];
                $error_messages = array_merge($error_messages, $result['errors']);
                break;
        }
    }
}

function importFromCuelinks($connect, $settings) {
    $count = 0;
    $errors = [];
    
    try {
        // Cuelinks API endpoint for deals/coupons
        $url = "https://api.cuelinks.com/v1/deals";
        $headers = [
            'Authorization: Bearer ' . $settings['api_key'],
            'Content-Type: application/json',
            'Accept: application/json'
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code == 200) {
            $data = json_decode($response, true);
            if (isset($data['data']) && is_array($data['data'])) {
                foreach ($data['data'] as $deal) {
                    if (importCouponFromData($connect, $deal, 'cuelinks')) {
                        $count++;
                    }
                }
            } elseif (isset($data['deals']) && is_array($data['deals'])) {
                foreach ($data['deals'] as $deal) {
                    if (importCouponFromData($connect, $deal, 'cuelinks')) {
                        $count++;
                    }
                }
            }
        } else {
            $errors[] = "Cuelinks API error: HTTP $http_code - " . $response;
        }
    } catch (Exception $e) {
        $errors[] = "Cuelinks import error: " . $e->getMessage();
    }
    
    return ['count' => $count, 'errors' => $errors];
}

function importFromINRDeals($connect, $settings) {
    $count = 0;
    $errors = [];
    
    try {
        // INR Deals API endpoint
        $url = "https://api.inrdeals.com/v1/coupons";
        $headers = [
            'Authorization: Bearer ' . $settings['api_key'],
            'Content-Type: application/json',
            'Accept: application/json'
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code == 200) {
            $data = json_decode($response, true);
            if (isset($data['data']) && is_array($data['data'])) {
                foreach ($data['data'] as $coupon) {
                    if (importCouponFromData($connect, $coupon, 'inrdeals')) {
                        $count++;
                    }
                }
            } elseif (isset($data['coupons']) && is_array($data['coupons'])) {
                foreach ($data['coupons'] as $coupon) {
                    if (importCouponFromData($connect, $coupon, 'inrdeals')) {
                        $count++;
                    }
                }
            }
        } else {
            $errors[] = "INR Deals API error: HTTP $http_code - " . $response;
        }
    } catch (Exception $e) {
        $errors[] = "INR Deals import error: " . $e->getMessage();
    }
    
    return ['count' => $count, 'errors' => $errors];
}

function importFromAdmitad($connect, $settings) {
    $count = 0;
    $errors = [];
    
    try {
        // First get access token using Base64 authentication
        $token_url = "https://api.admitad.com/token/";
        
        // Create Base64 encoded credentials
        $credentials = base64_encode($settings['api_key'] . ':' . $settings['api_secret']);
        
        $token_data = [
            'grant_type' => 'client_credentials',
            'scope' => 'coupons'
        ];
        
        $headers = [
            'Authorization: Basic ' . $credentials,
            'Content-Type: application/x-www-form-urlencoded'
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $token_url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($token_data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $token_response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code == 200) {
            $token_data = json_decode($token_response, true);
            if (isset($token_data['access_token'])) {
                $access_token = $token_data['access_token'];
                
                // Now get coupons using proper Admitad API endpoint
                $coupons_url = "https://api.admitad.com/coupons/";
                $headers = [
                    'Authorization: Bearer ' . $access_token,
                    'Content-Type: application/json'
                ];
                
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, $coupons_url);
                curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                curl_setopt($ch, CURLOPT_TIMEOUT, 30);
                
                $response = curl_exec($ch);
                $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                curl_close($ch);
                
                if ($http_code == 200) {
                    $data = json_decode($response, true);
                    if (isset($data['results']) && is_array($data['results'])) {
                        foreach ($data['results'] as $coupon) {
                            if (importCouponFromData($connect, $coupon, 'admitad')) {
                                $count++;
                            }
                        }
                    }
                } else {
                    $errors[] = "Admitad coupons API error: HTTP $http_code";
                }
            } else {
                $errors[] = "Failed to get Admitad access token";
            }
        } else {
            $errors[] = "Admitad token API error: HTTP $http_code";
        }
    } catch (Exception $e) {
        $errors[] = "Admitad import error: " . $e->getMessage();
    }
    
    return ['count' => $count, 'errors' => $errors];
}

function importFromCJ($connect, $settings) {
    $count = 0;
    $errors = [];
    
    try {
        // Commission Junction API endpoint for product search
        $url = "https://product-search.api.cj.com/v3/product-search";
        $headers = [
            'Authorization: Bearer ' . $settings['api_key'],
            'Content-Type: application/json',
            'Accept: application/json'
        ];
        
        // Add query parameters
        $params = [
            'website-id' => $settings['api_secret'], // Using api_secret as publisher_id
            'format' => 'json',
            'page-number' => 1,
            'records-per-page' => 100
        ];
        
        $url .= '?' . http_build_query($params);
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code == 200) {
            $data = json_decode($response, true);
            if (isset($data['products']) && is_array($data['products'])) {
                foreach ($data['products'] as $product) {
                    if (importCouponFromData($connect, $product, 'cj')) {
                        $count++;
                    }
                }
            } elseif (isset($data['links']) && is_array($data['links'])) {
                foreach ($data['links'] as $link) {
                    if (importCouponFromData($connect, $link, 'cj')) {
                        $count++;
                    }
                }
            }
        } else {
            $errors[] = "CJ API error: HTTP $http_code - " . $response;
        }
    } catch (Exception $e) {
        $errors[] = "CJ import error: " . $e->getMessage();
    }
    
    return ['count' => $count, 'errors' => $errors];
}

function importCouponFromData($connect, $data, $source) {
    try {
        // Extract common fields based on source
        $title = '';
        $description = '';
        $code = '';
        $link = '';
        $image_url = '';
        $store_name = '';
        $expiry = null;
        
        switch ($source) {
            case 'cuelinks':
                $title = $data['title'] ?? $data['deal_title'] ?? '';
                $description = $data['description'] ?? $data['deal_description'] ?? '';
                $code = $data['coupon_code'] ?? $data['code'] ?? '';
                $link = $data['affiliate_url'] ?? $data['url'] ?? '';
                $image_url = $data['image_url'] ?? $data['image'] ?? '';
                $store_name = $data['store_name'] ?? $data['merchant_name'] ?? '';
                $expiry = $data['expiry_date'] ?? $data['expires_at'] ?? null;
                break;
                
            case 'inrdeals':
                $title = $data['title'] ?? $data['coupon_title'] ?? '';
                $description = $data['description'] ?? $data['coupon_description'] ?? '';
                $code = $data['code'] ?? $data['coupon_code'] ?? '';
                $link = $data['url'] ?? $data['affiliate_url'] ?? '';
                $image_url = $data['image'] ?? $data['image_url'] ?? '';
                $store_name = $data['store'] ?? $data['merchant'] ?? '';
                $expiry = $data['expires'] ?? $data['expiry_date'] ?? null;
                break;
                
            case 'admitad':
                $title = $data['name'] ?? $data['title'] ?? '';
                $description = $data['description'] ?? $data['short_description'] ?? '';
                $code = $data['promocode'] ?? $data['code'] ?? '';
                $link = $data['goto_link'] ?? $data['url'] ?? '';
                $image_url = $data['image'] ?? $data['image_url'] ?? '';
                $store_name = $data['campaign']['name'] ?? $data['merchant_name'] ?? '';
                $expiry = $data['date_end'] ?? $data['expires_at'] ?? null;
                break;
                
            case 'cj':
                $title = $data['link-name'] ?? $data['name'] ?? '';
                $description = $data['description'] ?? $data['short-description'] ?? '';
                $code = $data['promo-code'] ?? $data['code'] ?? '';
                $link = $data['clickUrl'] ?? $data['url'] ?? '';
                $image_url = $data['image-url'] ?? $data['image'] ?? '';
                $store_name = $data['advertiser-name'] ?? $data['merchant-name'] ?? '';
                $expiry = $data['expiry-date'] ?? $data['expires-at'] ?? null;
                break;
        }
        
        if (empty($title) || empty($link)) {
            return false;
        }
        
        // Check if coupon already exists
        $stmt = $connect->prepare("SELECT coupon_id FROM coupons WHERE coupon_title = ? AND coupon_link = ?");
        $stmt->execute([$title, $link]);
        if ($stmt->fetch()) {
            return false; // Already exists
        }
        
        // Get or create store
        $store_id = getOrCreateStore($connect, $store_name, $link, $image_url);
        
        // Download and save image
        $image_filename = '';
        if (!empty($image_url)) {
            $image_filename = downloadAndSaveImage($image_url, $title);
        }
        
        // Insert coupon
        $stmt = $connect->prepare("INSERT INTO coupons (coupon_id, coupon_title, coupon_tagline, coupon_description, coupon_code, coupon_link, coupon_image, store_id, coupon_featured, coupon_status, coupon_expire) VALUES (null, ?, ?, ?, ?, ?, ?, ?, 0, 1, ?)");
        
        return $stmt->execute([
            $title,
            $description,
            $description,
            $code,
            $link,
            $image_filename,
            $store_id,
            $expiry
        ]);
        
    } catch (Exception $e) {
        return false;
    }
}

function getOrCreateStore($connect, $store_name, $store_url, $image_url) {
    if (empty($store_name)) {
        return null;
    }
    
    // Check if store exists
    $stmt = $connect->prepare("SELECT store_id FROM stores WHERE store_title = ?");
    $stmt->execute([$store_name]);
    $store = $stmt->fetch();
    
    if ($store) {
        return $store['store_id'];
    }
    
    // Create new store
    $slug = strtolower(str_replace(' ', '-', $store_name));
    $slug = preg_replace('/[^a-z0-9\-]/', '', $slug);
    
    // Download store image
    $store_image = '';
    if (!empty($image_url)) {
        $store_image = downloadAndSaveImage($image_url, $store_name . '_store');
    }
    
    $stmt = $connect->prepare("INSERT INTO stores (store_id, store_title, store_slug, store_description, store_seotitle, store_seodescription, store_featured, store_status, store_image, store_url) VALUES (null, ?, ?, '', ?, '', 0, 1, ?, ?)");
    
    if ($stmt->execute([$store_name, $slug, $store_name, $store_image, $store_url])) {
        return $connect->lastInsertId();
    }
    
    return null;
}

function downloadAndSaveImage($url, $filename) {
    try {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        
        $image_data = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code == 200 && $image_data) {
            $extension = pathinfo(parse_url($url, PHP_URL_PATH), PATHINFO_EXTENSION);
            if (empty($extension)) {
                $extension = 'jpg';
            }
            
            $filename = preg_replace('/[^a-zA-Z0-9\-_]/', '', $filename);
            $filename = $filename . '_' . time() . '.' . $extension;
            $filepath = '../../images/' . $filename;
            
            if (file_put_contents($filepath, $image_data)) {
                return $filename;
            }
        }
    } catch (Exception $e) {
        // Ignore image download errors
    }
    
    return '';
}

// Get available programs (all enabled programs, regardless of API key status)
$stmt = $connect->prepare("SELECT * FROM affiliate_programs WHERE status = 1");
$stmt->execute();
$available_programs = $stmt->fetchAll(PDO::FETCH_ASSOC);

require '../views/header.view.php';
require '../views/affiliate_import.view.php';
require '../views/footer.view.php';

} else {
    header('Location: ./login.php');
}
?>


require '../../config.php';
require '../admin_config.php';
require '../functions.php';

if (isset($_SESSION['user_email'])){

$connect = connect($database);

if (!isAdmin($connect) && !isAgent($connect)) {
    header('Location: ../../index.php');
    exit;
}

if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['import_from_affiliate'])) {
    $program = cleardata($_POST['affiliate_program']);
    $import_count = 0;
    $error_messages = [];
    
    // Get program settings
    $stmt = $connect->prepare("SELECT * FROM affiliate_programs WHERE program_name = ? AND status = 1");
    $stmt->execute([$program]);
    $program_settings = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Check if program has API key configured
    if (empty($program_settings['api_key'])) {
        $error_messages = ["❌ " . $program . " API key is not configured. Please configure the API credentials first."];
    }
    
    if (!$program_settings) {
        $error_messages[] = "Program not found or not enabled.";
    } else {
        switch ($program) {
            case 'cuelinks':
                $result = importFromCuelinks($connect, $program_settings);
                $import_count = $result['count'];
                $error_messages = array_merge($error_messages, $result['errors']);
                break;
                
            case 'inrdeals':
                $result = importFromINRDeals($connect, $program_settings);
                $import_count = $result['count'];
                $error_messages = array_merge($error_messages, $result['errors']);
                break;
                
            case 'admitad':
                $result = importFromAdmitad($connect, $program_settings);
                $import_count = $result['count'];
                $error_messages = array_merge($error_messages, $result['errors']);
                break;
                
            case 'cj':
                $result = importFromCJ($connect, $program_settings);
                $import_count = $result['count'];
                $error_messages = array_merge($error_messages, $result['errors']);
                break;
        }
    }
}

function importFromCuelinks($connect, $settings) {
    $count = 0;
    $errors = [];
    
    try {
        // Cuelinks API endpoint for deals/coupons
        $url = "https://api.cuelinks.com/v1/deals";
        $headers = [
            'Authorization: Bearer ' . $settings['api_key'],
            'Content-Type: application/json',
            'Accept: application/json'
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code == 200) {
            $data = json_decode($response, true);
            if (isset($data['data']) && is_array($data['data'])) {
                foreach ($data['data'] as $deal) {
                    if (importCouponFromData($connect, $deal, 'cuelinks')) {
                        $count++;
                    }
                }
            } elseif (isset($data['deals']) && is_array($data['deals'])) {
                foreach ($data['deals'] as $deal) {
                    if (importCouponFromData($connect, $deal, 'cuelinks')) {
                        $count++;
                    }
                }
            }
        } else {
            $errors[] = "Cuelinks API error: HTTP $http_code - " . $response;
        }
    } catch (Exception $e) {
        $errors[] = "Cuelinks import error: " . $e->getMessage();
    }
    
    return ['count' => $count, 'errors' => $errors];
}

function importFromINRDeals($connect, $settings) {
    $count = 0;
    $errors = [];
    
    try {
        // INR Deals API endpoint
        $url = "https://api.inrdeals.com/v1/coupons";
        $headers = [
            'Authorization: Bearer ' . $settings['api_key'],
            'Content-Type: application/json',
            'Accept: application/json'
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code == 200) {
            $data = json_decode($response, true);
            if (isset($data['data']) && is_array($data['data'])) {
                foreach ($data['data'] as $coupon) {
                    if (importCouponFromData($connect, $coupon, 'inrdeals')) {
                        $count++;
                    }
                }
            } elseif (isset($data['coupons']) && is_array($data['coupons'])) {
                foreach ($data['coupons'] as $coupon) {
                    if (importCouponFromData($connect, $coupon, 'inrdeals')) {
                        $count++;
                    }
                }
            }
        } else {
            $errors[] = "INR Deals API error: HTTP $http_code - " . $response;
        }
    } catch (Exception $e) {
        $errors[] = "INR Deals import error: " . $e->getMessage();
    }
    
    return ['count' => $count, 'errors' => $errors];
}

function importFromAdmitad($connect, $settings) {
    $count = 0;
    $errors = [];
    
    try {
        // First get access token using Base64 authentication
        $token_url = "https://api.admitad.com/token/";
        
        // Create Base64 encoded credentials
        $credentials = base64_encode($settings['api_key'] . ':' . $settings['api_secret']);
        
        $token_data = [
            'grant_type' => 'client_credentials',
            'scope' => 'coupons'
        ];
        
        $headers = [
            'Authorization: Basic ' . $credentials,
            'Content-Type: application/x-www-form-urlencoded'
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $token_url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($token_data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $token_response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code == 200) {
            $token_data = json_decode($token_response, true);
            if (isset($token_data['access_token'])) {
                $access_token = $token_data['access_token'];
                
                // Now get coupons using proper Admitad API endpoint
                $coupons_url = "https://api.admitad.com/coupons/";
                $headers = [
                    'Authorization: Bearer ' . $access_token,
                    'Content-Type: application/json'
                ];
                
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, $coupons_url);
                curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                curl_setopt($ch, CURLOPT_TIMEOUT, 30);
                
                $response = curl_exec($ch);
                $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                curl_close($ch);
                
                if ($http_code == 200) {
                    $data = json_decode($response, true);
                    if (isset($data['results']) && is_array($data['results'])) {
                        foreach ($data['results'] as $coupon) {
                            if (importCouponFromData($connect, $coupon, 'admitad')) {
                                $count++;
                            }
                        }
                    }
                } else {
                    $errors[] = "Admitad coupons API error: HTTP $http_code";
                }
            } else {
                $errors[] = "Failed to get Admitad access token";
            }
        } else {
            $errors[] = "Admitad token API error: HTTP $http_code";
        }
    } catch (Exception $e) {
        $errors[] = "Admitad import error: " . $e->getMessage();
    }
    
    return ['count' => $count, 'errors' => $errors];
}

function importFromCJ($connect, $settings) {
    $count = 0;
    $errors = [];
    
    try {
        // Commission Junction API endpoint for product search
        $url = "https://product-search.api.cj.com/v3/product-search";
        $headers = [
            'Authorization: Bearer ' . $settings['api_key'],
            'Content-Type: application/json',
            'Accept: application/json'
        ];
        
        // Add query parameters
        $params = [
            'website-id' => $settings['api_secret'], // Using api_secret as publisher_id
            'format' => 'json',
            'page-number' => 1,
            'records-per-page' => 100
        ];
        
        $url .= '?' . http_build_query($params);
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code == 200) {
            $data = json_decode($response, true);
            if (isset($data['products']) && is_array($data['products'])) {
                foreach ($data['products'] as $product) {
                    if (importCouponFromData($connect, $product, 'cj')) {
                        $count++;
                    }
                }
            } elseif (isset($data['links']) && is_array($data['links'])) {
                foreach ($data['links'] as $link) {
                    if (importCouponFromData($connect, $link, 'cj')) {
                        $count++;
                    }
                }
            }
        } else {
            $errors[] = "CJ API error: HTTP $http_code - " . $response;
        }
    } catch (Exception $e) {
        $errors[] = "CJ import error: " . $e->getMessage();
    }
    
    return ['count' => $count, 'errors' => $errors];
}

function importCouponFromData($connect, $data, $source) {
    try {
        // Extract common fields based on source
        $title = '';
        $description = '';
        $code = '';
        $link = '';
        $image_url = '';
        $store_name = '';
        $expiry = null;
        
        switch ($source) {
            case 'cuelinks':
                $title = $data['title'] ?? $data['deal_title'] ?? '';
                $description = $data['description'] ?? $data['deal_description'] ?? '';
                $code = $data['coupon_code'] ?? $data['code'] ?? '';
                $link = $data['affiliate_url'] ?? $data['url'] ?? '';
                $image_url = $data['image_url'] ?? $data['image'] ?? '';
                $store_name = $data['store_name'] ?? $data['merchant_name'] ?? '';
                $expiry = $data['expiry_date'] ?? $data['expires_at'] ?? null;
                break;
                
            case 'inrdeals':
                $title = $data['title'] ?? $data['coupon_title'] ?? '';
                $description = $data['description'] ?? $data['coupon_description'] ?? '';
                $code = $data['code'] ?? $data['coupon_code'] ?? '';
                $link = $data['url'] ?? $data['affiliate_url'] ?? '';
                $image_url = $data['image'] ?? $data['image_url'] ?? '';
                $store_name = $data['store'] ?? $data['merchant'] ?? '';
                $expiry = $data['expires'] ?? $data['expiry_date'] ?? null;
                break;
                
            case 'admitad':
                $title = $data['name'] ?? $data['title'] ?? '';
                $description = $data['description'] ?? $data['short_description'] ?? '';
                $code = $data['promocode'] ?? $data['code'] ?? '';
                $link = $data['goto_link'] ?? $data['url'] ?? '';
                $image_url = $data['image'] ?? $data['image_url'] ?? '';
                $store_name = $data['campaign']['name'] ?? $data['merchant_name'] ?? '';
                $expiry = $data['date_end'] ?? $data['expires_at'] ?? null;
                break;
                
            case 'cj':
                $title = $data['link-name'] ?? $data['name'] ?? '';
                $description = $data['description'] ?? $data['short-description'] ?? '';
                $code = $data['promo-code'] ?? $data['code'] ?? '';
                $link = $data['clickUrl'] ?? $data['url'] ?? '';
                $image_url = $data['image-url'] ?? $data['image'] ?? '';
                $store_name = $data['advertiser-name'] ?? $data['merchant-name'] ?? '';
                $expiry = $data['expiry-date'] ?? $data['expires-at'] ?? null;
                break;
        }
        
        if (empty($title) || empty($link)) {
            return false;
        }
        
        // Check if coupon already exists
        $stmt = $connect->prepare("SELECT coupon_id FROM coupons WHERE coupon_title = ? AND coupon_link = ?");
        $stmt->execute([$title, $link]);
        if ($stmt->fetch()) {
            return false; // Already exists
        }
        
        // Get or create store
        $store_id = getOrCreateStore($connect, $store_name, $link, $image_url);
        
        // Download and save image
        $image_filename = '';
        if (!empty($image_url)) {
            $image_filename = downloadAndSaveImage($image_url, $title);
        }
        
        // Insert coupon
        $stmt = $connect->prepare("INSERT INTO coupons (coupon_id, coupon_title, coupon_tagline, coupon_description, coupon_code, coupon_link, coupon_image, store_id, coupon_featured, coupon_status, coupon_expire) VALUES (null, ?, ?, ?, ?, ?, ?, ?, 0, 1, ?)");
        
        return $stmt->execute([
            $title,
            $description,
            $description,
            $code,
            $link,
            $image_filename,
            $store_id,
            $expiry
        ]);
        
    } catch (Exception $e) {
        return false;
    }
}

function getOrCreateStore($connect, $store_name, $store_url, $image_url) {
    if (empty($store_name)) {
        return null;
    }
    
    // Check if store exists
    $stmt = $connect->prepare("SELECT store_id FROM stores WHERE store_title = ?");
    $stmt->execute([$store_name]);
    $store = $stmt->fetch();
    
    if ($store) {
        return $store['store_id'];
    }
    
    // Create new store
    $slug = strtolower(str_replace(' ', '-', $store_name));
    $slug = preg_replace('/[^a-z0-9\-]/', '', $slug);
    
    // Download store image
    $store_image = '';
    if (!empty($image_url)) {
        $store_image = downloadAndSaveImage($image_url, $store_name . '_store');
    }
    
    $stmt = $connect->prepare("INSERT INTO stores (store_id, store_title, store_slug, store_description, store_seotitle, store_seodescription, store_featured, store_status, store_image, store_url) VALUES (null, ?, ?, '', ?, '', 0, 1, ?, ?)");
    
    if ($stmt->execute([$store_name, $slug, $store_name, $store_image, $store_url])) {
        return $connect->lastInsertId();
    }
    
    return null;
}

function downloadAndSaveImage($url, $filename) {
    try {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        
        $image_data = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code == 200 && $image_data) {
            $extension = pathinfo(parse_url($url, PHP_URL_PATH), PATHINFO_EXTENSION);
            if (empty($extension)) {
                $extension = 'jpg';
            }
            
            $filename = preg_replace('/[^a-zA-Z0-9\-_]/', '', $filename);
            $filename = $filename . '_' . time() . '.' . $extension;
            $filepath = '../../images/' . $filename;
            
            if (file_put_contents($filepath, $image_data)) {
                return $filename;
            }
        }
    } catch (Exception $e) {
        // Ignore image download errors
    }
    
    return '';
}

// Get available programs (all enabled programs, regardless of API key status)
$stmt = $connect->prepare("SELECT * FROM affiliate_programs WHERE status = 1");
$stmt->execute();
$available_programs = $stmt->fetchAll(PDO::FETCH_ASSOC);

require '../views/header.view.php';
require '../views/affiliate_import.view.php';
require '../views/footer.view.php';

} else {
    header('Location: ./login.php');
}
?>
