<?php
/**
 * Slick Mega Menu Walker
 *
 * @package Slick
 * @since 1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Slick Mega Menu Walker
 */
class Slick_Mega_Menu_Walker extends Walker_Nav_Menu {
    
    /**
     * Starts the list before the elements are added.
     */
    public function start_lvl( &$output, $depth = 0, $args = null ) {
        $indent = str_repeat( "\t", $depth );
        
        if ( $depth === 0 ) {
            // Check if parent item is a mega menu
            $parent_item = $this->get_parent_item( $output );
            $is_mega_menu = is_object( $parent_item ) && isset( $parent_item->mega_menu ) && $parent_item->mega_menu;
            
            if ( $is_mega_menu ) {
                $mega_menu_style = isset( $parent_item->mega_menu_style ) ? $parent_item->mega_menu_style : 'style1';
                $mega_menu_columns = isset( $parent_item->mega_menu_columns ) ? $parent_item->mega_menu_columns : '4';
                
                $output .= "\n$indent<div class=\"mega-menu-container mega-menu-{$mega_menu_style} columns-{$mega_menu_columns} clearfix\"><div class=\"container\"><div class=\"mega-menu-content\">\n";
            } else {
                $output .= "\n$indent<ul class=\"sub-menu\">\n";
            }
        } else {
            $output .= "\n$indent<ul class=\"sub-menu\">\n";
        }
    }
    
    /**
     * Ends the list of after the elements are added.
     */
    public function end_lvl( &$output, $depth = 0, $args = null ) {
        $indent = str_repeat( "\t", $depth );
        
        if ( $depth === 0 ) {
            // Check if parent item is a mega menu
            $parent_item = $this->get_parent_item( $output );
            $is_mega_menu = is_object( $parent_item ) && isset( $parent_item->mega_menu ) && $parent_item->mega_menu;
            
            if ( $is_mega_menu ) {
                $output .= "$indent</div></div></div>\n";
            } else {
                $output .= "$indent</ul>\n";
            }
        } else {
            $output .= "$indent</ul>\n";
        }
    }
    
    /**
     * Starts the element output.
     */
    public function start_el( &$output, $item, $depth = 0, $args = null, $id = 0 ) {
        $indent = ( $depth ) ? str_repeat( "\t", $depth ) : '';
        
        $classes = empty( $item->classes ) ? array() : (array) $item->classes;
        $classes[] = 'menu-item-' . $item->ID;
        
        // Add mega menu class
        if ( $item->mega_menu ) {
            $classes[] = 'mega-menu-item';
        }
        
        // Add column class for mega menu items
        if ( $depth === 1 && $this->is_mega_menu_item( $item ) ) {
            $parent_item = $this->get_parent_item( $output );
            $mega_menu_columns = isset( $parent_item->mega_menu_columns ) ? $parent_item->mega_menu_columns : '4';
            $classes[] = 'mega-menu-column';
            $classes[] = 'column-' . $mega_menu_columns;
        }
        
        $class_names = join( ' ', apply_filters( 'nav_menu_css_class', array_filter( $classes ), $item, $args, $depth ) );
        $class_names = $class_names ? ' class="' . esc_attr( $class_names ) . '"' : '';
        
        $id = 'menu-item-' . $item->ID;
        $id = $id ? ' id="' . esc_attr( $id ) . '"' : '';
        
        $output .= $indent . '<li' . $id . $class_names . '>';
        
        $atts = array();
        $atts['title']  = ! empty( $item->attr_title ) ? $item->attr_title : '';
        $atts['target'] = ! empty( $item->target ) ? $item->target : '';
        $atts['rel']    = ! empty( $item->xfn ) ? $item->xfn : '';
        $atts['href']   = ! empty( $item->url ) ? $item->url : '';
        $atts['class']  = '';
        
        // Add class for top-level menu items
        if ( $depth === 0 ) {
            $atts['class'] .= 'menu-link';
        }
        
        // Add class for mega menu items
        if ( $item->mega_menu ) {
            $atts['class'] .= ' mega-menu-link';
        }
        
        $atts = apply_filters( 'nav_menu_link_attributes', $atts, $item, $args, $depth );
        
        $attributes = '';
        foreach ( $atts as $attr => $value ) {
            if ( ! empty( $value ) ) {
                $value = ( 'href' === $attr ) ? esc_url( $value ) : esc_attr( $value );
                $attributes .= ' ' . $attr . '="' . $value . '"';
            }
        }
        
        $title = apply_filters( 'the_title', $item->title, $item->ID );
        $title = apply_filters( 'nav_menu_item_title', $title, $item, $args, $depth );
        
        $item_output = $args->before;
        $item_output .= '<a' . $attributes . '>';
        $item_output .= $args->link_before . $title . $args->link_after;
        
        // Add dropdown indicator for parent items
        if ( in_array( 'menu-item-has-children', $classes ) ) {
            $item_output .= ' <span class="dropdown-indicator"><i class="fas fa-chevron-down"></i></span>';
        }
        
        $item_output .= '</a>';
        $item_output .= $args->after;
        
        // Add featured image for mega menu items
        if ( $depth === 1 && $this->is_mega_menu_item( $item ) && has_post_thumbnail( $item->object_id ) ) {
            $item_output .= '<div class="mega-menu-featured-image">' . get_the_post_thumbnail( $item->object_id, 'medium' ) . '</div>';
        }
        
        // Add description for mega menu items
        if ( $depth === 1 && $this->is_mega_menu_item( $item ) && ! empty( $item->description ) ) {
            $item_output .= '<div class="mega-menu-description">' . $item->description . '</div>';
        }
        
        $output .= apply_filters( 'walker_nav_menu_start_el', $item_output, $item, $depth, $args );
    }
    
    /**
     * Ends the element output, if needed.
     */
    public function end_el( &$output, $item, $depth = 0, $args = null ) {
        $output .= "</li>\n";
    }
    
    /**
     * Check if the current item is part of a mega menu.
     */
    protected function is_mega_menu_item( $item ) {
        $parent_id = $item->menu_item_parent;
        
        if ( ! $parent_id ) {
            return false;
        }
        
        $parent_item = $this->get_menu_item( $parent_id );
        
        if ( ! $parent_item ) {
            return false;
        }
        
        return isset( $parent_item->mega_menu ) && $parent_item->mega_menu;
    }
    
    /**
     * Get a menu item by ID.
     */
    protected function get_menu_item( $item_id ) {
        $menu_items = wp_get_nav_menu_items( $this->current_menu );
        
        if ( ! $menu_items ) {
            return null;
        }
        
        foreach ( $menu_items as $menu_item ) {
            if ( $menu_item->ID == $item_id ) {
                return $menu_item;
            }
        }
        
        return null;
    }
    
    /**
     * Get the parent menu item from the output.
     */
    protected function get_parent_item( &$output ) {
        $menu_items = wp_get_nav_menu_items( $this->current_menu );
        
        if ( ! $menu_items ) {
            return null;
        }
        
        // Get the last opened menu item
        $last_opened_item = end( $this->parent_items );
        
        if ( $last_opened_item ) {
            return $this->get_menu_item( $last_opened_item );
        }
        
        return null;
    }
}
?>
