<?php
/**
 * 2FA Setup Status Checker
 * Verifies that all components are properly installed
 */

require 'core.php';

// Check if admin is logged in
if (!isAdmin()) {
    die('Access denied. Admin login required.');
}

// Check database table
$table_exists = false;
try {
    $result = $connect->query("SHOW TABLES LIKE 'verification_codes'");
    $table_exists = $result->rowCount() > 0;
} catch (Exception $e) {
    $table_exists = false;
}

// Check email template
$template_exists = false;
try {
    $stmt = $connect->prepare("SELECT * FROM emailtemplates WHERE email_id = 3");
    $stmt->execute();
    $template_exists = $stmt->rowCount() > 0;
} catch (Exception $e) {
    $template_exists = false;
}

// Check SMTP settings
$smtp_configured = false;
try {
    $settings = $connect->query("SELECT * FROM settings")->fetch();
    $smtp_configured = !empty($settings['st_smtphost']) && 
                       !empty($settings['st_smtpemail']) && 
                       !empty($settings['st_smtppassword']);
} catch (Exception $e) {
    $smtp_configured = false;
}

// Check required files
$files_check = [
    'controllers/verify_2fa.php' => file_exists('controllers/verify_2fa.php'),
    'controllers/resend_2fa.php' => file_exists('controllers/resend_2fa.php'),
    'views/verify-otp.view.php' => file_exists('views/verify-otp.view.php'),
    'assets/js/auth-handler.js' => file_exists('assets/js/auth-handler.js')
];

$all_files_exist = !in_array(false, $files_check);

// Overall status
$setup_complete = $table_exists && $template_exists && $smtp_configured && $all_files_exist;

?>
<!DOCTYPE html>
<html>
<head>
    <title>2FA Setup Status</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            max-width: 900px;
            margin: 50px auto;
            padding: 20px;
            background: #f5f5f5;
        }
        .container {
            background: white;
            padding: 30px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #333;
            border-bottom: 2px solid #667eea;
            padding-bottom: 10px;
        }
        .status-card {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 8px;
            margin: 15px 0;
            border-left: 4px solid #ddd;
        }
        .status-card.success {
            border-left-color: #28a745;
            background: #d4edda;
        }
        .status-card.error {
            border-left-color: #dc3545;
            background: #f8d7da;
        }
        .status-card h3 {
            margin-top: 0;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        .check-item {
            padding: 10px;
            margin: 5px 0;
            border-radius: 4px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        .check-item.pass {
            background: #d4edda;
            color: #155724;
        }
        .check-item.fail {
            background: #f8d7da;
            color: #721c24;
        }
        .icon {
            font-size: 20px;
            font-weight: bold;
        }
        .overall-status {
            text-align: center;
            padding: 30px;
            margin: 20px 0;
            border-radius: 8px;
            font-size: 24px;
            font-weight: bold;
        }
        .overall-status.complete {
            background: #d4edda;
            color: #155724;
        }
        .overall-status.incomplete {
            background: #fff3cd;
            color: #856404;
        }
        .btn {
            display: inline-block;
            background: #667eea;
            color: white;
            padding: 12px 30px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            text-decoration: none;
            font-size: 16px;
            font-weight: bold;
            margin: 5px;
        }
        .btn:hover {
            background: #5568d3;
        }
        .btn-success {
            background: #28a745;
        }
        .btn-success:hover {
            background: #218838;
        }
        .instructions {
            background: #d1ecf1;
            color: #0c5460;
            padding: 15px;
            border-radius: 4px;
            border: 1px solid #bee5eb;
            margin: 20px 0;
        }
        .instructions ol {
            margin: 10px 0;
            padding-left: 20px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔐 2FA Setup Status Checker</h1>
        
        <div class="overall-status <?php echo $setup_complete ? 'complete' : 'incomplete'; ?>">
            <?php if ($setup_complete): ?>
                ✅ 2FA Setup Complete!
            <?php else: ?>
                ⚠️ 2FA Setup Incomplete
            <?php endif; ?>
        </div>

        <div class="status-card <?php echo $table_exists ? 'success' : 'error'; ?>">
            <h3>
                <span class="icon"><?php echo $table_exists ? '✅' : '❌'; ?></span>
                Database Table
            </h3>
            <div class="check-item <?php echo $table_exists ? 'pass' : 'fail'; ?>">
                <span class="icon"><?php echo $table_exists ? '✓' : '✗'; ?></span>
                <span>verification_codes table</span>
            </div>
            <?php if (!$table_exists): ?>
                <div class="instructions">
                    <strong>Action Required:</strong> Import database_2fa_update.sql
                </div>
            <?php endif; ?>
        </div>

        <div class="status-card <?php echo $template_exists ? 'success' : 'error'; ?>">
            <h3>
                <span class="icon"><?php echo $template_exists ? '✅' : '❌'; ?></span>
                Email Template
            </h3>
            <div class="check-item <?php echo $template_exists ? 'pass' : 'fail'; ?>">
                <span class="icon"><?php echo $template_exists ? '✓' : '✗'; ?></span>
                <span>2FA Verification Code template (ID: 3)</span>
            </div>
            <?php if (!$template_exists): ?>
                <div class="instructions">
                    <strong>Action Required:</strong> Import database_2fa_update.sql
                </div>
            <?php endif; ?>
        </div>

        <div class="status-card <?php echo $smtp_configured ? 'success' : 'error'; ?>">
            <h3>
                <span class="icon"><?php echo $smtp_configured ? '✅' : '❌'; ?></span>
                SMTP Configuration
            </h3>
            <div class="check-item <?php echo !empty($settings['st_smtphost']) ? 'pass' : 'fail'; ?>">
                <span class="icon"><?php echo !empty($settings['st_smtphost']) ? '✓' : '✗'; ?></span>
                <span>SMTP Host: <?php echo !empty($settings['st_smtphost']) ? htmlspecialchars($settings['st_smtphost']) : 'Not configured'; ?></span>
            </div>
            <div class="check-item <?php echo !empty($settings['st_smtpemail']) ? 'pass' : 'fail'; ?>">
                <span class="icon"><?php echo !empty($settings['st_smtpemail']) ? '✓' : '✗'; ?></span>
                <span>SMTP Email: <?php echo !empty($settings['st_smtpemail']) ? htmlspecialchars($settings['st_smtpemail']) : 'Not configured'; ?></span>
            </div>
            <div class="check-item <?php echo !empty($settings['st_smtppassword']) ? 'pass' : 'fail'; ?>">
                <span class="icon"><?php echo !empty($settings['st_smtppassword']) ? '✓' : '✗'; ?></span>
                <span>SMTP Password: <?php echo !empty($settings['st_smtppassword']) ? 'Configured' : 'Not configured'; ?></span>
            </div>
            <?php if (!$smtp_configured): ?>
                <div class="instructions">
                    <strong>Action Required:</strong> Configure SMTP settings in Admin Panel → Settings
                </div>
            <?php endif; ?>
        </div>

        <div class="status-card <?php echo $all_files_exist ? 'success' : 'error'; ?>">
            <h3>
                <span class="icon"><?php echo $all_files_exist ? '✅' : '❌'; ?></span>
                Required Files
            </h3>
            <?php foreach ($files_check as $file => $exists): ?>
                <div class="check-item <?php echo $exists ? 'pass' : 'fail'; ?>">
                    <span class="icon"><?php echo $exists ? '✓' : '✗'; ?></span>
                    <span><?php echo htmlspecialchars($file); ?></span>
                </div>
            <?php endforeach; ?>
        </div>

        <?php if ($setup_complete): ?>
            <div style="text-align: center; margin-top: 30px;">
                <a href="test_email.php" class="btn btn-success">📧 Test Email Configuration</a>
                <a href="<?php echo SITE_URL; ?>/signin.php" class="btn">🔐 Test 2FA Login</a>
                <a href="cleanup_verification_codes.php" class="btn">🧹 Cleanup Old Codes</a>
            </div>
        <?php else: ?>
            <div class="instructions">
                <strong>Setup Instructions:</strong>
                <ol>
                    <li>Import database_2fa_update.sql into your database</li>
                    <li>Configure SMTP settings in Admin Panel</li>
                    <li>Test email configuration using test_email.php</li>
                    <li>Test 2FA login flow</li>
                </ol>
            </div>
        <?php endif; ?>

        <div style="text-align: center; margin-top: 20px;">
            <a href="<?php echo SITE_URL; ?>/admin" class="btn">← Back to Admin</a>
            <a href="<?php echo $_SERVER['PHP_SELF']; ?>" class="btn">🔄 Refresh Status</a>
        </div>

        <div style="margin-top: 30px; padding-top: 20px; border-top: 1px solid #ddd; color: #666; font-size: 14px;">
            <strong>📚 Documentation:</strong>
            <ul>
                <li><a href="2FA_SETUP_INSTRUCTIONS.md" target="_blank">Complete Setup Instructions</a></li>
                <li><a href="2FA_ADMIN_GUIDE.md" target="_blank">Admin Quick Reference Guide</a></li>
            </ul>
        </div>
    </div>
</body>
</html>
