<?php
/**
 * Enhanced Coupon Card Component
 * With Featured/Popular Labels and Countdown Timer
 * 
 * Usage:
 * Include this in your coupon listing pages
 */

// Example coupon data structure
// $coupon = [
//     'coupon_id' => 1,
//     'coupon_title' => 'Get 50% Off',
//     'coupon_image' => 'coupon_123.jpg',
//     'coupon_featured' => 1,
//     'coupon_exclusive' => 1,
//     'coupon_expire' => '2025-01-15 23:59:59',
//     'coupon_code' => 'SAVE50',
//     'store_title' => 'Example Store'
// ];

function renderCouponCard($coupon) {
    // Determine label type
    $labelType = '';
    $labelText = '';
    $labelIcon = '';
    
    if (isset($coupon['coupon_exclusive']) && $coupon['coupon_exclusive'] == 1) {
        $labelType = 'exclusive';
        $labelText = 'Exclusive';
        $labelIcon = '⭐';
    } elseif (isset($coupon['coupon_featured']) && $coupon['coupon_featured'] == 1) {
        $labelType = 'featured';
        $labelText = 'Featured';
        $labelIcon = '🔥';
    } elseif (isset($coupon['coupon_popular']) && $coupon['coupon_popular'] == 1) {
        $labelType = 'popular';
        $labelText = 'Popular';
        $labelIcon = '💎';
    }
    
    // Check if countdown should be shown
    $showCountdown = false;
    $expiryDate = '';
    if (!empty($coupon['coupon_expire']) && $coupon['coupon_expire'] != '0000-00-00 00:00:00') {
        $expiryTimestamp = strtotime($coupon['coupon_expire']);
        if ($expiryTimestamp > time()) {
            $showCountdown = true;
            $expiryDate = date('c', $expiryTimestamp); // ISO 8601 format
        }
    }
    
    // Get image URL (with CDN support)
    $imageUrl = getCouponImageUrl($GLOBALS['connect'], $coupon);
    
    ?>
    <div class="coupon-card">
        <?php if ($labelType): ?>
        <div class="coupon-label <?php echo $labelType; ?>">
            <i><?php echo $labelIcon; ?></i>
            <?php echo $labelText; ?>
        </div>
        <?php endif; ?>
        
        <div class="coupon-image">
            <img src="<?php echo htmlspecialchars($imageUrl); ?>" 
                 alt="<?php echo htmlspecialchars($coupon['coupon_title']); ?>"
                 loading="lazy">
        </div>
        
        <div class="coupon-content">
            <h3 class="coupon-title">
                <?php echo htmlspecialchars($coupon['coupon_title']); ?>
            </h3>
            
            <?php if (!empty($coupon['store_title'])): ?>
            <p class="coupon-store">
                <?php echo htmlspecialchars($coupon['store_title']); ?>
            </p>
            <?php endif; ?>
            
            <?php if (!empty($coupon['coupon_code'])): ?>
            <div class="coupon-code">
                <code><?php echo htmlspecialchars($coupon['coupon_code']); ?></code>
            </div>
            <?php endif; ?>
        </div>
        
        <?php if ($showCountdown): ?>
        <div class="coupon-countdown" data-countdown data-expiry="<?php echo $expiryDate; ?>">
            <span class="countdown-icon">⚡</span>
            <span class="countdown-label">Loading...</span>
        </div>
        <?php endif; ?>
    </div>
    <?php
}

// Example usage in your template:
/*
<link rel="stylesheet" href="assets/css/coupon-card-enhancements.css">
<script src="assets/js/coupon-countdown.js"></script>

<div class="coupons-grid">
    <?php foreach ($coupons as $coupon): ?>
        <?php renderCouponCard($coupon); ?>
    <?php endforeach; ?>
</div>
*/
?>
