/**
 * Modern Footer Newsletter
 * Handles subscription with validation and feedback
 */

class ModernNewsletterFooter {
    constructor(formId) {
        this.form = document.getElementById(formId);
        if (!this.form) return;
        
        this.input = this.form.querySelector('.modern-newsletter-input');
        this.button = this.form.querySelector('.modern-newsletter-submit');
        this.messageContainer = this.form.querySelector('.modern-newsletter-message');
        
        this.init();
    }
    
    init() {
        this.form.addEventListener('submit', (e) => {
            e.preventDefault();
            this.handleSubmit();
        });
        
        // Real-time validation
        this.input.addEventListener('input', () => {
            this.clearMessage();
        });
    }
    
    async handleSubmit() {
        const email = this.input.value.trim();
        
        // Validate email
        if (!email) {
            this.showError('Please enter your email address');
            return;
        }
        
        if (!this.isValidEmail(email)) {
            this.showError('Please enter a valid email address');
            return;
        }
        
        // Show loading state
        this.setLoading(true);
        
        try {
            const response = await fetch('/subscribe_newsletter.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({ email: email })
            });
            
            const data = await response.json();
            
            if (data.success) {
                this.showSuccess(data.message || 'Thank you for subscribing!');
                this.input.value = '';
            } else {
                this.showError(data.message || 'Subscription failed. Please try again.');
            }
        } catch (error) {
            this.showError('An error occurred. Please try again later.');
        } finally {
            this.setLoading(false);
        }
    }
    
    isValidEmail(email) {
        const regex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return regex.test(email);
    }
    
    setLoading(loading) {
        this.button.disabled = loading;
        this.button.textContent = loading ? 'Subscribing...' : 'Subscribe';
    }
    
    showSuccess(message) {
        this.showMessage(message, 'success');
    }
    
    showError(message) {
        this.showMessage(message, 'error');
    }
    
    showMessage(message, type) {
        if (!this.messageContainer) {
            this.messageContainer = document.createElement('div');
            this.messageContainer.className = 'modern-newsletter-message';
            this.form.appendChild(this.messageContainer);
        }
        
        this.messageContainer.textContent = message;
        this.messageContainer.className = `modern-newsletter-message modern-newsletter-${type}`;
        this.messageContainer.style.display = 'block';
        
        // Auto-hide after 5 seconds
        setTimeout(() => {
            this.clearMessage();
        }, 5000);
    }
    
    clearMessage() {
        if (this.messageContainer) {
            this.messageContainer.style.display = 'none';
        }
    }
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', () => {
    new ModernNewsletterFooter('modernNewsletterForm');
});
